<?php
class Auth {
    private $conn;
    private $table_name = "users";

    public function __construct($db) {
        $this->conn = $db;
    }

    public function validateToken($token) {
        // First, check if token is valid
        if (empty($token) || !is_string($token)) {
            error_log("Invalid token format provided");
            return false;
        }
        
        try {
            // Use a more robust query with error handling
            $query = "SELECT id, email, username 
                     FROM " . $this->table_name . " 
                     WHERE remember_token = :token";
                     
            $stmt = $this->conn->prepare($query);
            $stmt->bindParam(":token", $token);
            $stmt->execute();
            
            // Debug: Log the query and row count
            error_log("Token validation query executed. Rows found: " . $stmt->rowCount());
            
            if($stmt->rowCount() > 0) {
                return $stmt->fetch(PDO::FETCH_ASSOC);
            }
            
            // If no user found with this token, log for debugging
            error_log("No user found with token: " . substr($token, 0, 10) . "...");
            return false;
            
        } catch(PDOException $e) {
            // Log the error
            error_log("Database error during token validation: " . $e->getMessage());
            return false;
        }
    }
    
    // Helper method to verify if a token exists in the database
    public function tokenExists($token) {
        try {
            $query = "SELECT COUNT(*) FROM " . $this->table_name . " WHERE remember_token = :token";
            $stmt = $this->conn->prepare($query);
            $stmt->bindParam(":token", $token);
            $stmt->execute();
            
            return ($stmt->fetchColumn() > 0);
        } catch(PDOException $e) {
            error_log("Error checking token existence: " . $e->getMessage());
            return false;
        }
    }
}
?>