<?php
// Enable error reporting for debugging
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Headers
header("Access-Control-Allow-Origin: *");
header("Content-Type: application/json; charset=UTF-8");
header("Access-Control-Allow-Methods: GET, OPTIONS");
header("Access-Control-Max-Age: 3600");
header("Access-Control-Allow-Headers: Content-Type, Access-Control-Allow-Headers, Authorization, X-Requested-With");

// Handle preflight requests
if ($_SERVER['REQUEST_METHOD'] == 'OPTIONS') {
    http_response_code(200);
    exit();
}

try {
    // Include database and auth class
    require_once '../config/Database.php';
    require_once '../middleware/Auth.php';

    // Initialize database
    $database = new Database();
    $db = $database->getConnection();

    if (!$db) {
        throw new Exception("Database connection failed");
    }

    // Initialize Auth
    $auth = new Auth($db);

    // Get the token from headers
    $token = null;
    $allHeaders = getallheaders();
    
    foreach ($allHeaders as $name => $value) {
        if (strtolower($name) === 'authorization') {
            if (preg_match('/Bearer\s+(.+)/', $value, $matches)) {
                $token = $matches[1];
            } else {
                $token = $value;
            }
            break;
        }
    }

    // Validate token
    if (!$token) {
        http_response_code(401);
        echo json_encode(array(
            "status" => false,
            "message" => "Access token is required"
        ));
        exit();
    }

    $user = $auth->validateToken($token);
    if (!$user) {
        http_response_code(401);
        echo json_encode(array(
            "status" => false,
            "message" => "Invalid or expired token"
        ));
        exit();
    }

    // Get request method and report type
    $method = $_SERVER['REQUEST_METHOD'];
    $reportType = isset($_GET['type']) ? $_GET['type'] : '';

    if ($method !== 'GET') {
        http_response_code(405);
        echo json_encode(array(
            "status" => false,
            "message" => "Only GET method is allowed"
        ));
        exit();
    }

    // Route requests
    switch ($reportType) {
        case 'monthly-summary':
            getMonthlyFinancialSummary($db, $user);
            break;
        case 'fee-collection-report':
            getFeeCollectionReport($db, $user);
            break;
        case 'expense-analysis':
            getExpenseAnalysis($db, $user);
            break;
        case 'income-analysis':
            getIncomeAnalysis($db, $user);
            break;
        case 'outstanding-fees':
            getOutstandingFeesReport($db, $user);
            break;
        case 'cash-flow':
            getCashFlowReport($db, $user);
            break;
        case 'profit-loss':
            getProfitLossReport($db, $user);
            break;
        case 'student-payment-history':
            getStudentPaymentHistory($db, $user);
            break;
        default:
            http_response_code(404);
            echo json_encode(array(
                "status" => false,
                "message" => "Report type not found. Available types: monthly-summary, fee-collection-report, expense-analysis, income-analysis, outstanding-fees, cash-flow, profit-loss, student-payment-history"
            ));
            break;
    }

} catch (Exception $e) {
    error_log("Financial Reports API Error: " . $e->getMessage());
    http_response_code(500);
    echo json_encode(array(
        "status" => false,
        "message" => "Internal server error: " . $e->getMessage()
    ));
}

/**
 * Get monthly financial summary
 */
function getMonthlyFinancialSummary($db, $user) {
    try {
        $year = isset($_GET['year']) ? (int)$_GET['year'] : date('Y');
        $month = isset($_GET['month']) ? (int)$_GET['month'] : date('n');
        
        $startDate = sprintf('%04d-%02d-01', $year, $month);
        $endDate = date('Y-m-t', strtotime($startDate));
        
        // Get monthly income
        $incomeQuery = "SELECT 
                           COALESCE(SUM(amount), 0) as total_income,
                           COUNT(*) as income_transactions
                       FROM incomes 
                       WHERE date >= :start_date AND date <= :end_date";
        $incomeStmt = $db->prepare($incomeQuery);
        $incomeStmt->bindParam(':start_date', $startDate);
        $incomeStmt->bindParam(':end_date', $endDate);
        $incomeStmt->execute();
        $incomeData = $incomeStmt->fetch(PDO::FETCH_ASSOC);
        
        // Get monthly expenses
        $expenseQuery = "SELECT 
                            COALESCE(SUM(amount), 0) as total_expenses,
                            COUNT(*) as expense_transactions
                        FROM expenses 
                        WHERE date >= :start_date AND date <= :end_date";
        $expenseStmt = $db->prepare($expenseQuery);
        $expenseStmt->bindParam(':start_date', $startDate);
        $expenseStmt->bindParam(':end_date', $endDate);
        $expenseStmt->execute();
        $expenseData = $expenseStmt->fetch(PDO::FETCH_ASSOC);
        
        // Get monthly fee collection
        $feeQuery = "SELECT 
                        COALESCE(SUM(amount_paid), 0) as total_fee_collected,
                        COUNT(*) as fee_transactions
                    FROM student_fee_records 
                    WHERE payment_date >= :start_date AND payment_date <= :end_date 
                    AND status = 'paid'";
        $feeStmt = $db->prepare($feeQuery);
        $feeStmt->bindParam(':start_date', $startDate);
        $feeStmt->bindParam(':end_date', $endDate);
        $feeStmt->execute();
        $feeData = $feeStmt->fetch(PDO::FETCH_ASSOC);
        
        // Get expense breakdown by category
        $categoryQuery = "SELECT 
                             tc.name as category,
                             COALESCE(SUM(e.amount), 0) as amount,
                             COUNT(e.id) as transactions
                         FROM expenses e
                         LEFT JOIN transaction_categories tc ON e.transaction_category_id = tc.id
                         WHERE e.date >= :start_date AND e.date <= :end_date
                         GROUP BY tc.id, tc.name
                         ORDER BY amount DESC";
        $categoryStmt = $db->prepare($categoryQuery);
        $categoryStmt->bindParam(':start_date', $startDate);
        $categoryStmt->bindParam(':end_date', $endDate);
        $categoryStmt->execute();
        $categoryBreakdown = $categoryStmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Calculate totals
        $totalIncome = (float)$incomeData['total_income'] + (float)$feeData['total_fee_collected'];
        $totalExpenses = (float)$expenseData['total_expenses'];
        $netProfit = $totalIncome - $totalExpenses;
        
        http_response_code(200);
        echo json_encode(array(
            "status" => true,
            "message" => "Monthly financial summary retrieved successfully",
            "data" => array(
                "period" => array(
                    "year" => $year,
                    "month" => $month,
                    "month_name" => date('F', mktime(0, 0, 0, $month, 1)),
                    "start_date" => $startDate,
                    "end_date" => $endDate
                ),
                "summary" => array(
                    "total_income" => $totalIncome,
                    "total_expenses" => $totalExpenses,
                    "net_profit" => $netProfit,
                    "profit_margin" => $totalIncome > 0 ? round(($netProfit / $totalIncome) * 100, 2) : 0
                ),
                "breakdown" => array(
                    "other_income" => array(
                        "amount" => (float)$incomeData['total_income'],
                        "transactions" => (int)$incomeData['income_transactions']
                    ),
                    "fee_collection" => array(
                        "amount" => (float)$feeData['total_fee_collected'],
                        "transactions" => (int)$feeData['fee_transactions']
                    ),
                    "expenses" => array(
                        "amount" => (float)$expenseData['total_expenses'],
                        "transactions" => (int)$expenseData['expense_transactions']
                    )
                ),
                "expense_categories" => $categoryBreakdown
            )
        ));
        
    } catch (Exception $e) {
        error_log("Error getting monthly summary: " . $e->getMessage());
        http_response_code(500);
        echo json_encode(array(
            "status" => false,
            "message" => "Error retrieving monthly financial summary"
        ));
    }
}

/**
 * Get fee collection report
 */
function getFeeCollectionReport($db, $user) {
    try {
        $academicSession = isset($_GET['academic_session']) ? $_GET['academic_session'] : null;
        $batchId = isset($_GET['batch_id']) ? $_GET['batch_id'] : null;
        $dateFrom = isset($_GET['date_from']) ? $_GET['date_from'] : null;
        $dateTo = isset($_GET['date_to']) ? $_GET['date_to'] : null;
        
        $whereClause = "WHERE 1=1";
        $params = array();
        
        if ($academicSession) {
            $whereClause .= " AND sr.academic_session_id = :academic_session";
            $params[':academic_session'] = $academicSession;
        }
        
        if ($batchId) {
            $whereClause .= " AND sr.batch_id = :batch_id";
            $params[':batch_id'] = $batchId;
        }
        
        if ($dateFrom) {
            $whereClause .= " AND sfr.payment_date >= :date_from";
            $params[':date_from'] = $dateFrom;
        }
        
        if ($dateTo) {
            $whereClause .= " AND sfr.payment_date <= :date_to";
            $params[':date_to'] = $dateTo;
        }
        
        // Get collection summary by batch
        $summaryQuery = "SELECT 
                            b.name as batch_name,
                            c.name as course_name,
                            COUNT(DISTINCT sr.student_id) as total_students,
                            COUNT(CASE WHEN sfr.status = 'paid' THEN 1 END) as paid_count,
                            COUNT(CASE WHEN sfr.status = 'unpaid' THEN 1 END) as unpaid_count,
                            COALESCE(SUM(CASE WHEN sfr.status = 'paid' THEN sfr.amount_paid END), 0) as collected_amount,
                            COALESCE(SUM(CASE WHEN sfr.status = 'unpaid' THEN fi.amount END), 0) as pending_amount
                        FROM student_records sr
                        JOIN batches b ON sr.batch_id = b.id
                        JOIN courses c ON b.course_id = c.id
                        LEFT JOIN student_fee_records sfr ON sr.id = sfr.student_record_id
                        LEFT JOIN fee_installments fi ON sfr.fee_installment_id = fi.id
                        $whereClause
                        GROUP BY b.id, b.name, c.name
                        ORDER BY b.name";
        
        $summaryStmt = $db->prepare($summaryQuery);
        foreach ($params as $key => $value) {
            $summaryStmt->bindValue($key, $value);
        }
        $summaryStmt->execute();
        $batchSummary = $summaryStmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Get detailed collection data
        $detailQuery = "SELECT 
                           s.first_name, s.last_name, s.contact_number,
                           sr.roll_number,
                           b.name as batch_name,
                           fi.title as installment_title,
                           fi.due_date,
                           fi.amount as installment_amount,
                           sfr.status,
                           sfr.amount_paid,
                           sfr.payment_date
                       FROM student_records sr
                       JOIN students s ON sr.student_id = s.id
                       JOIN batches b ON sr.batch_id = b.id
                       LEFT JOIN student_fee_records sfr ON sr.id = sfr.student_record_id
                       LEFT JOIN fee_installments fi ON sfr.fee_installment_id = fi.id
                       $whereClause
                       ORDER BY b.name, s.first_name, fi.due_date";
        
        $detailStmt = $db->prepare($detailQuery);
        foreach ($params as $key => $value) {
            $detailStmt->bindValue($key, $value);
        }
        $detailStmt->execute();
        $detailData = $detailStmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Calculate overall totals
        $totalCollected = 0;
        $totalPending = 0;
        $totalStudents = 0;
        
        foreach ($batchSummary as $batch) {
            $totalCollected += (float)$batch['collected_amount'];
            $totalPending += (float)$batch['pending_amount'];
            $totalStudents += (int)$batch['total_students'];
        }
        
        http_response_code(200);
        echo json_encode(array(
            "status" => true,
            "message" => "Fee collection report retrieved successfully",
            "data" => array(
                "summary" => array(
                    "total_students" => $totalStudents,
                    "total_collected" => $totalCollected,
                    "total_pending" => $totalPending,
                    "collection_percentage" => ($totalCollected + $totalPending) > 0 ? 
                        round(($totalCollected / ($totalCollected + $totalPending)) * 100, 2) : 0
                ),
                "batch_summary" => $batchSummary,
                "detailed_records" => $detailData
            )
        ));
        
    } catch (Exception $e) {
        error_log("Error getting fee collection report: " . $e->getMessage());
        http_response_code(500);
        echo json_encode(array(
            "status" => false,
            "message" => "Error retrieving fee collection report"
        ));
    }
}

/**
 * Get expense analysis
 */
function getExpenseAnalysis($db, $user) {
    try {
        $dateFrom = isset($_GET['date_from']) ? $_GET['date_from'] : date('Y-m-01');
        $dateTo = isset($_GET['date_to']) ? $_GET['date_to'] : date('Y-m-d');
        
        // Get expense trends (monthly)
        $trendsQuery = "SELECT 
                           DATE_FORMAT(date_of_expense, '%Y-%m') as month,
                           COALESCE(SUM(amount), 0) as total_amount,
                           COUNT(*) as transaction_count
                       FROM expenses
                       WHERE date_of_expense >= :date_from AND date_of_expense <= :date_to
                       GROUP BY DATE_FORMAT(date_of_expense, '%Y-%m')
                       ORDER BY month";
        
        $trendsStmt = $db->prepare($trendsQuery);
        $trendsStmt->bindParam(':date_from', $dateFrom);
        $trendsStmt->bindParam(':date_to', $dateTo);
        $trendsStmt->execute();
        $trends = $trendsStmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Get category-wise breakdown
        $categoryQuery = "SELECT 
                             tc.name as category,
                             COALESCE(SUM(e.amount), 0) as total_amount,
                             COUNT(e.id) as transaction_count,
                             AVG(e.amount) as average_amount
                         FROM expenses e
                         LEFT JOIN transaction_categories tc ON e.transaction_category_id = tc.id
                         WHERE e.date_of_expense >= :date_from AND e.date_of_expense <= :date_to
                         GROUP BY tc.id, tc.name
                         ORDER BY total_amount DESC";
        
        $categoryStmt = $db->prepare($categoryQuery);
        $categoryStmt->bindParam(':date_from', $dateFrom);
        $categoryStmt->bindParam(':date_to', $dateTo);
        $categoryStmt->execute();
        $categories = $categoryStmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Get top expenses
        $topExpensesQuery = "SELECT 
                                e.description,
                                amount,
                                date_of_expense,
                                tc.name as category
                            FROM expenses e
                            LEFT JOIN transaction_categories tc ON e.transaction_category_id = tc.id
                            WHERE e.date_of_expense >= :date_from AND e.date_of_expense <= :date_to
                            ORDER BY amount DESC
                            LIMIT 10";
        
        $topExpensesStmt = $db->prepare($topExpensesQuery);
        $topExpensesStmt->bindParam(':date_from', $dateFrom);
        $topExpensesStmt->bindParam(':date_to', $dateTo);
        $topExpensesStmt->execute();
        $topExpenses = $topExpensesStmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Calculate total and average
        $totalAmount = 0;
        $totalTransactions = 0;
        
        foreach ($categories as $category) {
            $totalAmount += (float)$category['total_amount'];
            $totalTransactions += (int)$category['transaction_count'];
        }
        
        $averageExpense = $totalTransactions > 0 ? $totalAmount / $totalTransactions : 0;
        
        http_response_code(200);
        echo json_encode(array(
            "status" => true,
            "message" => "Expense analysis retrieved successfully",
            "data" => array(
                "period" => array(
                    "from" => $dateFrom,
                    "to" => $dateTo
                ),
                "summary" => array(
                    "total_amount" => $totalAmount,
                    "total_transactions" => $totalTransactions,
                    "average_expense" => round($averageExpense, 2)
                ),
                "monthly_trends" => $trends,
                "category_breakdown" => $categories,
                "top_expenses" => $topExpenses
            )
        ));
        
    } catch (Exception $e) {
        error_log("Error getting expense analysis: " . $e->getMessage());
        http_response_code(500);
        echo json_encode(array(
            "status" => false,
            "message" => "Error retrieving expense analysis"
        ));
    }
}

/**
 * Get income analysis
 */
function getIncomeAnalysis($db, $user) {
    try {
        $dateFrom = isset($_GET['date_from']) ? $_GET['date_from'] : date('Y-m-01');
        $dateTo = isset($_GET['date_to']) ? $_GET['date_to'] : date('Y-m-d');
        
        // Get income trends (monthly)
        $trendsQuery = "SELECT 
                           DATE_FORMAT(date_of_income, '%Y-%m') as month,
                           COALESCE(SUM(amount), 0) as total_amount,
                           COUNT(*) as transaction_count
                       FROM incomes
                       WHERE date_of_income >= :date_from AND date_of_income <= :date_to
                       GROUP BY DATE_FORMAT(date_of_income, '%Y-%m')
                       ORDER BY month";
        
        $trendsStmt = $db->prepare($trendsQuery);
        $trendsStmt->bindParam(':date_from', $dateFrom);
        $trendsStmt->bindParam(':date_to', $dateTo);
        $trendsStmt->execute();
        $trends = $trendsStmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Get fee collection trends
        $feeTrendsQuery = "SELECT 
                              DATE_FORMAT(payment_date, '%Y-%m') as month,
                              COALESCE(SUM(amount_paid), 0) as total_amount,
                              COUNT(*) as transaction_count
                          FROM student_fee_records
                          WHERE payment_date >= :date_from AND payment_date <= :date_to
                          AND status = 'paid'
                          GROUP BY DATE_FORMAT(payment_date, '%Y-%m')
                          ORDER BY month";
        
        $feeTrendsStmt = $db->prepare($feeTrendsQuery);
        $feeTrendsStmt->bindParam(':date_from', $dateFrom);
        $feeTrendsStmt->bindParam(':date_to', $dateTo);
        $feeTrendsStmt->execute();
        $feeTrends = $feeTrendsStmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Get category-wise income breakdown
        $categoryQuery = "SELECT 
                             tc.name as category,
                             COALESCE(SUM(i.amount), 0) as total_amount,
                             COUNT(i.id) as transaction_count
                         FROM incomes i
                         LEFT JOIN transaction_categories tc ON i.transaction_category_id = tc.id
                         WHERE i.date_of_income >= :date_from AND i.date_of_income <= :date_to
                         GROUP BY tc.id, tc.name
                         ORDER BY total_amount DESC";
        
        $categoryStmt = $db->prepare($categoryQuery);
        $categoryStmt->bindParam(':date_from', $dateFrom);
        $categoryStmt->bindParam(':date_to', $dateTo);
        $categoryStmt->execute();
        $categories = $categoryStmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Calculate totals
        $totalOtherIncome = 0;
        $totalFeeIncome = 0;
        
        foreach ($categories as $category) {
            $totalOtherIncome += (float)$category['total_amount'];
        }
        
        foreach ($feeTrends as $fee) {
            $totalFeeIncome += (float)$fee['total_amount'];
        }
        
        $totalIncome = $totalOtherIncome + $totalFeeIncome;
        
        http_response_code(200);
        echo json_encode(array(
            "status" => true,
            "message" => "Income analysis retrieved successfully",
            "data" => array(
                "period" => array(
                    "from" => $dateFrom,
                    "to" => $dateTo
                ),
                "summary" => array(
                    "total_income" => $totalIncome,
                    "fee_income" => $totalFeeIncome,
                    "other_income" => $totalOtherIncome,
                    "fee_percentage" => $totalIncome > 0 ? round(($totalFeeIncome / $totalIncome) * 100, 2) : 0
                ),
                "monthly_trends" => array(
                    "other_income" => $trends,
                    "fee_collection" => $feeTrends
                ),
                "income_categories" => $categories
            )
        ));
        
    } catch (Exception $e) {
        error_log("Error getting income analysis: " . $e->getMessage());
        http_response_code(500);
        echo json_encode(array(
            "status" => false,
            "message" => "Error retrieving income analysis"
        ));
    }
}

/**
 * Get outstanding fees report
 */
function getOutstandingFeesReport($db, $user) {
    try {
        $batchId = isset($_GET['batch_id']) ? $_GET['batch_id'] : null;
        $daysOverdue = isset($_GET['days_overdue']) ? (int)$_GET['days_overdue'] : 0;
        
        $whereClause = "WHERE sfr.status = 'unpaid'";
        $params = array();
        
        if ($batchId) {
            $whereClause .= " AND sr.batch_id = :batch_id";
            $params[':batch_id'] = $batchId;
        }
        
        if ($daysOverdue > 0) {
            $whereClause .= " AND fi.due_date < DATE_SUB(CURDATE(), INTERVAL :days_overdue DAY)";
            $params[':days_overdue'] = $daysOverdue;
        }
        
        $query = "SELECT 
                     s.first_name, s.last_name, s.contact_number, s.email,
                     sr.roll_number,
                     b.name as batch_name,
                     c.name as course_name,
                     fi.title as installment_title,
                     fi.due_date,
                     fi.amount as outstanding_amount,
                     DATEDIFF(CURDATE(), fi.due_date) as days_overdue
                 FROM student_fee_records sfr
                 JOIN student_records sr ON sfr.student_record_id = sr.id
                 JOIN students s ON sr.student_id = s.id
                 JOIN batches b ON sr.batch_id = b.id
                 JOIN courses c ON b.course_id = c.id
                 JOIN fee_installments fi ON sfr.fee_installment_id = fi.id
                 $whereClause
                 ORDER BY fi.due_date ASC, s.first_name ASC";
        
        $stmt = $db->prepare($query);
        foreach ($params as $key => $value) {
            $stmt->bindValue($key, $value);
        }
        $stmt->execute();
        
        $outstandingFees = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Calculate summary
        $totalAmount = 0;
        $totalStudents = count($outstandingFees);
        $overdueCount = 0;
        
        foreach ($outstandingFees as &$record) {
            $totalAmount += (float)$record['outstanding_amount'];
            if ((int)$record['days_overdue'] > 0) {
                $overdueCount++;
            }
        }
        
        http_response_code(200);
        echo json_encode(array(
            "status" => true,
            "message" => "Outstanding fees report retrieved successfully",
            "data" => array(
                "summary" => array(
                    "total_students" => $totalStudents,
                    "total_outstanding_amount" => $totalAmount,
                    "overdue_count" => $overdueCount,
                    "current_count" => $totalStudents - $overdueCount
                ),
                "outstanding_fees" => $outstandingFees
            )
        ));
        
    } catch (Exception $e) {
        error_log("Error getting outstanding fees report: " . $e->getMessage());
        http_response_code(500);
        echo json_encode(array(
            "status" => false,
            "message" => "Error retrieving outstanding fees report"
        ));
    }
}

/**
 * Get cash flow report
 */
function getCashFlowReport($db, $user) {
    try {
        $dateFrom = isset($_GET['date_from']) ? $_GET['date_from'] : date('Y-m-01');
        $dateTo = isset($_GET['date_to']) ? $_GET['date_to'] : date('Y-m-d');
        
        // Get daily cash flow
        $dailyQuery = "SELECT 
                          date,
                          'income' as type,
                          SUM(amount) as amount
                      FROM incomes
                      WHERE date >= :date_from AND date <= :date_to
                      GROUP BY date
                      
                      UNION ALL
                      
                      SELECT 
                          payment_date as date,
                          'fee_collection' as type,
                          SUM(amount_paid) as amount
                      FROM student_fee_records
                      WHERE payment_date >= :date_from2 AND payment_date <= :date_to2
                      AND status = 'paid'
                      GROUP BY payment_date
                      
                      UNION ALL
                      
                      SELECT 
                          date,
                          'expense' as type,
                          -SUM(amount) as amount
                      FROM expenses
                      WHERE date >= :date_from3 AND date <= :date_to3
                      GROUP BY date
                      
                      ORDER BY date";
        
        $dailyStmt = $db->prepare($dailyQuery);
        $dailyStmt->bindParam(':date_from', $dateFrom);
        $dailyStmt->bindParam(':date_to', $dateTo);
        $dailyStmt->bindParam(':date_from2', $dateFrom);
        $dailyStmt->bindParam(':date_to2', $dateTo);
        $dailyStmt->bindParam(':date_from3', $dateFrom);
        $dailyStmt->bindParam(':date_to3', $dateTo);
        $dailyStmt->execute();
        
        $dailyFlow = $dailyStmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Process daily flow to calculate running balance
        $processedFlow = array();
        $runningBalance = 0;
        $currentDate = null;
        $dayTotal = 0;
        
        foreach ($dailyFlow as $flow) {
            if ($currentDate !== $flow['date']) {
                if ($currentDate !== null) {
                    $runningBalance += $dayTotal;
                    $processedFlow[] = array(
                        'date' => $currentDate,
                        'net_flow' => $dayTotal,
                        'running_balance' => $runningBalance
                    );
                }
                $currentDate = $flow['date'];
                $dayTotal = 0;
            }
            $dayTotal += (float)$flow['amount'];
        }
        
        // Add the last day
        if ($currentDate !== null) {
            $runningBalance += $dayTotal;
            $processedFlow[] = array(
                'date' => $currentDate,
                'net_flow' => $dayTotal,
                'running_balance' => $runningBalance
            );
        }
        
        // Get period totals
        $totalIncome = 0;
        $totalExpenses = 0;
        $totalFeeCollection = 0;
        
        foreach ($dailyFlow as $flow) {
            if ($flow['type'] === 'income') {
                $totalIncome += (float)$flow['amount'];
            } elseif ($flow['type'] === 'fee_collection') {
                $totalFeeCollection += (float)$flow['amount'];
            } elseif ($flow['type'] === 'expense') {
                $totalExpenses += abs((float)$flow['amount']);
            }
        }
        
        $netCashFlow = $totalIncome + $totalFeeCollection - $totalExpenses;
        
        http_response_code(200);
        echo json_encode(array(
            "status" => true,
            "message" => "Cash flow report retrieved successfully",
            "data" => array(
                "period" => array(
                    "from" => $dateFrom,
                    "to" => $dateTo
                ),
                "summary" => array(
                    "total_inflow" => $totalIncome + $totalFeeCollection,
                    "total_outflow" => $totalExpenses,
                    "net_cash_flow" => $netCashFlow,
                    "income_breakdown" => array(
                        "other_income" => $totalIncome,
                        "fee_collection" => $totalFeeCollection
                    )
                ),
                "daily_flow" => $processedFlow
            )
        ));
        
    } catch (Exception $e) {
        error_log("Error getting cash flow report: " . $e->getMessage());
        http_response_code(500);
        echo json_encode(array(
            "status" => false,
            "message" => "Error retrieving cash flow report"
        ));
    }
}

/**
 * Get profit and loss report
 */
function getProfitLossReport($db, $user) {
    try {
        $dateFrom = isset($_GET['date_from']) ? $_GET['date_from'] : date('Y-01-01');
        $dateTo = isset($_GET['date_to']) ? $_GET['date_to'] : date('Y-m-d');
        
        // Get revenue (income + fee collection)
        $revenueQuery = "SELECT 
                            'Fee Collection' as source,
                            COALESCE(SUM(amount_paid), 0) as amount
                        FROM student_fee_records
                        WHERE payment_date >= :date_from AND payment_date <= :date_to
                        AND status = 'paid'
                        
                        UNION ALL
                        
                        SELECT 
                            COALESCE(tc.name, 'Other Income') as source,
                            COALESCE(SUM(i.amount), 0) as amount
                        FROM incomes i
                        LEFT JOIN transaction_categories tc ON i.transaction_category_id = tc.id
                        WHERE i.date >= :date_from2 AND i.date <= :date_to2
                        GROUP BY tc.id, tc.name";
        
        $revenueStmt = $db->prepare($revenueQuery);
        $revenueStmt->bindParam(':date_from', $dateFrom);
        $revenueStmt->bindParam(':date_to', $dateTo);
        $revenueStmt->bindParam(':date_from2', $dateFrom);
        $revenueStmt->bindParam(':date_to2', $dateTo);
        $revenueStmt->execute();
        $revenue = $revenueStmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Get expenses by category
        $expenseQuery = "SELECT 
                            COALESCE(tc.name, 'Uncategorized') as category,
                            COALESCE(SUM(e.amount), 0) as amount
                        FROM expenses e
                        LEFT JOIN transaction_categories tc ON e.transaction_category_id = tc.id
                        WHERE e.date >= :date_from AND e.date <= :date_to
                        GROUP BY tc.id, tc.name
                        ORDER BY amount DESC";
        
        $expenseStmt = $db->prepare($expenseQuery);
        $expenseStmt->bindParam(':date_from', $dateFrom);
        $expenseStmt->bindParam(':date_to', $dateTo);
        $expenseStmt->execute();
        $expenses = $expenseStmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Calculate totals
        $totalRevenue = 0;
        $totalExpenses = 0;
        
        foreach ($revenue as $rev) {
            $totalRevenue += (float)$rev['amount'];
        }
        
        foreach ($expenses as $exp) {
            $totalExpenses += (float)$exp['amount'];
        }
        
        $grossProfit = $totalRevenue;
        $netProfit = $totalRevenue - $totalExpenses;
        $profitMargin = $totalRevenue > 0 ? ($netProfit / $totalRevenue) * 100 : 0;
        
        http_response_code(200);
        echo json_encode(array(
            "status" => true,
            "message" => "Profit and loss report retrieved successfully",
            "data" => array(
                "period" => array(
                    "from" => $dateFrom,
                    "to" => $dateTo
                ),
                "revenue" => array(
                    "total" => $totalRevenue,
                    "breakdown" => $revenue
                ),
                "expenses" => array(
                    "total" => $totalExpenses,
                    "breakdown" => $expenses
                ),
                "profit_loss" => array(
                    "gross_profit" => $grossProfit,
                    "net_profit" => $netProfit,
                    "profit_margin_percentage" => round($profitMargin, 2),
                    "status" => $netProfit >= 0 ? 'profit' : 'loss'
                )
            )
        ));
        
    } catch (Exception $e) {
        error_log("Error getting profit and loss report: " . $e->getMessage());
        http_response_code(500);
        echo json_encode(array(
            "status" => false,
            "message" => "Error retrieving profit and loss report"
        ));
    }
}

/**
 * Get student payment history
 */
function getStudentPaymentHistory($db, $user) {
    try {
        $studentId = isset($_GET['student_id']) ? $_GET['student_id'] : null;
        
        if (!$studentId) {
            http_response_code(400);
            echo json_encode(array(
                "status" => false,
                "message" => "student_id parameter is required"
            ));
            return;
        }
        
        $query = "SELECT 
                     sfr.*,
                     fi.title as installment_title,
                     fi.due_date,
                     fi.amount as installment_amount,
                     sr.roll_number,
                     b.name as batch_name,
                     c.name as course_name,
                     acs.name as academic_session
                 FROM student_fee_records sfr
                 JOIN student_records sr ON sfr.student_record_id = sr.id
                 JOIN batches b ON sr.batch_id = b.id
                 JOIN courses c ON b.course_id = c.id
                 JOIN academic_sessions acs ON sr.academic_session_id = acs.id
                 LEFT JOIN fee_installments fi ON sfr.fee_installment_id = fi.id
                 WHERE sr.student_id = :student_id
                 ORDER BY fi.due_date DESC, sfr.created_at DESC";
        
        $stmt = $db->prepare($query);
        $stmt->bindParam(':student_id', $studentId);
        $stmt->execute();
        
        $paymentHistory = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Calculate summary
        $totalPaid = 0;
        $totalPending = 0;
        $paidCount = 0;
        $pendingCount = 0;
        
        foreach ($paymentHistory as $payment) {
            if ($payment['status'] === 'paid') {
                $totalPaid += (float)$payment['amount_paid'];
                $paidCount++;
            } else {
                $totalPending += (float)$payment['installment_amount'];
                $pendingCount++;
            }
        }
        
        http_response_code(200);
        echo json_encode(array(
            "status" => true,
            "message" => "Student payment history retrieved successfully",
            "data" => array(
                "student_id" => $studentId,
                "summary" => array(
                    "total_paid" => $totalPaid,
                    "total_pending" => $totalPending,
                    "paid_installments" => $paidCount,
                    "pending_installments" => $pendingCount,
                    "total_installments" => count($paymentHistory)
                ),
                "payment_history" => $paymentHistory
            )
        ));
        
    } catch (Exception $e) {
        error_log("Error getting student payment history: " . $e->getMessage());
        http_response_code(500);
        echo json_encode(array(
            "status" => false,
            "message" => "Error retrieving student payment history"
        ));
    }
}

?>