<?php
// Enable error reporting for debugging
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Headers
header("Access-Control-Allow-Origin: *");
header("Content-Type: application/json; charset=UTF-8");
header("Access-Control-Allow-Methods: GET, POST, PUT, DELETE, OPTIONS");
header("Access-Control-Max-Age: 3600");
header("Access-Control-Allow-Headers: Content-Type, Access-Control-Allow-Headers, Authorization, X-Requested-With");

// Handle preflight requests
if ($_SERVER['REQUEST_METHOD'] == 'OPTIONS') {
    http_response_code(200);
    exit();
}

// Debug: Log the raw request
error_log("Raw request received: " . file_get_contents('php://input'));

try {
    // Include database and auth class
    require_once '../config/Database.php';
    require_once '../middleware/Auth.php';

    // Initialize database
    $database = new Database();
    $db = $database->getConnection();

    // Check database connection
    if (!$db) {
        throw new Exception("Database connection failed");
    }

    // Initialize Auth
    $auth = new Auth($db);

    // Get the token from multiple possible sources
    $token = null;
    
    // 1. Try from getallheaders() first (case-insensitive)
    $allHeaders = getallheaders();
    error_log("All headers: " . print_r($allHeaders, true));
    
    foreach ($allHeaders as $name => $value) {
        if (strtolower($name) === 'authorization') {
            // Extract token - handle both "Bearer token" and raw token formats
            if (preg_match('/Bearer\s+(.+)/', $value, $matches)) {
                $token = $matches[1];
            } else {
                $token = $value;
            }
            error_log("Found token in getallheaders(): " . substr($token, 0, 10) . "...");
            break;
        }
    }
    
    // 2. Try from $_SERVER if still not found
    if (!$token) {
        // Apache specific
        if (isset($_SERVER['HTTP_AUTHORIZATION'])) {
            $authHeader = $_SERVER['HTTP_AUTHORIZATION'];
            
            if (preg_match('/Bearer\s+(.+)/', $authHeader, $matches)) {
                $token = $matches[1];
            } else {
                $token = $authHeader;
            }
            error_log("Found token in \$_SERVER['HTTP_AUTHORIZATION']: " . substr($token, 0, 10) . "...");
        }
        // Apache with mod_rewrite
        elseif (isset($_SERVER['REDIRECT_HTTP_AUTHORIZATION'])) {
            $authHeader = $_SERVER['REDIRECT_HTTP_AUTHORIZATION'];
            
            if (preg_match('/Bearer\s+(.+)/', $authHeader, $matches)) {
                $token = $matches[1];
            } else {
                $token = $authHeader;
            }
            error_log("Found token in \$_SERVER['REDIRECT_HTTP_AUTHORIZATION']: " . substr($token, 0, 10) . "...");
        }
        // Try to get raw headers
        else {
            foreach ($_SERVER as $key => $value) {
                if (substr($key, 0, 5) === 'HTTP_') {
                    $key = str_replace(' ', '-', ucwords(strtolower(str_replace('_', ' ', substr($key, 5)))));
                    if (strtolower($key) === 'authorization') {
                        if (preg_match('/Bearer\s+(.+)/', $value, $matches)) {
                            $token = $matches[1];
                        } else {
                            $token = $value;
                        }
                        error_log("Found token in \$_SERVER HTTP_ variable: " . substr($token, 0, 10) . "...");
                        break;
                    }
                }
            }
        }
    }
    
    // 3. Last resort: Look in GET parameters
    if (!$token && isset($_GET['token'])) {
        $token = $_GET['token'];
        error_log("Found token in GET parameter: " . substr($token, 0, 10) . "...");
    }

    // Validate token
    if (!$token) {
        http_response_code(401);
        echo json_encode(array(
            "status" => false,
            "message" => "No authorization token provided",
            "headers_received" => $allHeaders
        ));
        exit();
    }

    // Check if token exists in database before validation
    if (!$auth->tokenExists($token)) {
        error_log("Token does not exist in database: " . substr($token, 0, 10) . "...");
        http_response_code(401);
        echo json_encode(array(
            "status" => false,
            "message" => "Invalid token: not found in database"
        ));
        exit();
    }

    $user = $auth->validateToken($token);
    
    if (!$user) {
        http_response_code(401);
        echo json_encode(array(
            "status" => false,
            "message" => "Invalid or expired token"
        ));
        exit();
    }

    class CourseAndBatch {
        private $conn;
        
        public function __construct($db) {
            $this->conn = $db;
        }
        // Get course groups
public function getCourseGroupDetails($courseGroupId = null) {
    try {
        $query = "SELECT 
                    cg.id, 
                    cg.name, 
                    cg.position, 
                    cg.description,
                    acs.id as academic_session_id,
                    acs.name as academic_session_name,
                    acs.start_date as session_start_date,
                    acs.end_date as session_end_date,
                    acs.is_default as session_is_default
                FROM course_groups cg
                LEFT JOIN academic_sessions acs ON cg.academic_session_id = acs.id";
        
        if($courseGroupId) {
            $query .= " WHERE cg.id = :course_group_id";
        }
        
        $stmt = $this->conn->prepare($query);
        
        if($courseGroupId) {
            $stmt->bindParam(":course_group_id", $courseGroupId, PDO::PARAM_INT);
        }
        
        $stmt->execute();
        
        $result = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        if (!$result) {
            return array();
        }
        
        // Format dates
        foreach($result as &$row) {
            if(isset($row['session_start_date'])) {
                $row['session_start_date'] = date('Y-m-d', strtotime($row['session_start_date']));
            }
            if(isset($row['session_end_date'])) {
                $row['session_end_date'] = date('Y-m-d', strtotime($row['session_end_date']));
            }
        }
        
        return $result;
    } catch(PDOException $e) {
        throw new Exception("Error fetching course group details: " . $e->getMessage());
    }
}

// Create a course group
public function createCourseGroup($data) {
    try {
        // Validate required fields
        if(!isset($data->name) || !isset($data->academic_session_id)) {
            throw new Exception("Name and academic session ID are required fields");
        }
        
        $query = "INSERT INTO course_groups (
                    name, 
                    description, 
                    position,
                    academic_session_id,
                    created_at, 
                    updated_at
                  ) VALUES (
                    :name, 
                    :description, 
                    :position,
                    :academic_session_id,
                    NOW(), 
                    NOW()
                  )";
        
        $stmt = $this->conn->prepare($query);
        
        // Clean and bind data
        $name = htmlspecialchars(strip_tags($data->name));
        $description = isset($data->description) ? htmlspecialchars(strip_tags($data->description)) : null;
        $position = isset($data->position) ? intval($data->position) : 0;
        $academicSessionId = intval($data->academic_session_id);
        
        $stmt->bindParam(":name", $name);
        $stmt->bindParam(":description", $description);
        $stmt->bindParam(":position", $position);
        $stmt->bindParam(":academic_session_id", $academicSessionId);
        
        if($stmt->execute()) {
            $courseGroupId = $this->conn->lastInsertId();
            return array(
                "id" => $courseGroupId,
                "name" => $name,
                "description" => $description,
                "position" => $position,
                "academic_session_id" => $academicSessionId
            );
        }
        
        throw new Exception("Unknown error occurred when creating course group");
        
    } catch(PDOException $e) {
        throw new Exception("Database error: " . $e->getMessage());
    }
}

// Update a course group
public function updateCourseGroup($id, $data) {
    try {
        // First check if course group exists
        $courseGroupExists = $this->courseGroupExists($id);
        if(!$courseGroupExists) {
            throw new Exception("Course group with ID $id not found");
        }
        
        // Build query based on provided fields
        $updateFields = array();
        $params = array();
        
        if(isset($data->name)) {
            $updateFields[] = "name = :name";
            $params[':name'] = htmlspecialchars(strip_tags($data->name));
        }
        
        if(isset($data->description)) {
            $updateFields[] = "description = :description";
            $params[':description'] = htmlspecialchars(strip_tags($data->description));
        }
        
        if(isset($data->position)) {
            $updateFields[] = "position = :position";
            $params[':position'] = intval($data->position);
        }
        
        if(isset($data->academic_session_id)) {
            $updateFields[] = "academic_session_id = :academic_session_id";
            $params[':academic_session_id'] = intval($data->academic_session_id);
        }
        
        // Add updated_at timestamp
        $updateFields[] = "updated_at = NOW()";
        
        // If no fields to update, return early
        if(empty($updateFields)) {
            throw new Exception("No fields provided for update");
        }
        
        $query = "UPDATE course_groups SET " . implode(", ", $updateFields) . " WHERE id = :id";
        $params[':id'] = $id;
        
        $stmt = $this->conn->prepare($query);
        
        foreach($params as $param => $value) {
            $stmt->bindValue($param, $value);
        }
        
        if($stmt->execute()) {
            // Return updated course group data
            $updatedCourseGroup = $this->getCourseGroupDetails($id);
            return !empty($updatedCourseGroup) ? $updatedCourseGroup[0] : array();
        }
        
        throw new Exception("Unknown error occurred when updating course group");
        
    } catch(PDOException $e) {
        throw new Exception("Database error: " . $e->getMessage());
    }
}

// Delete a course group
public function deleteCourseGroup($id) {
    try {
        // First check if course group exists
        $courseGroupExists = $this->courseGroupExists($id);
        if(!$courseGroupExists) {
            throw new Exception("Course group with ID $id not found");
        }
        
        // Check if course group has associated courses
        $query = "SELECT COUNT(*) as course_count FROM courses WHERE course_group_id = :id";
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(":id", $id, PDO::PARAM_INT);
        $stmt->execute();
        $result = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if($result['course_count'] > 0) {
            throw new Exception("Cannot delete course group with ID $id because it has associated courses");
        }
        
        // Delete the course group
        $query = "DELETE FROM course_groups WHERE id = :id";
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(":id", $id, PDO::PARAM_INT);
        
        return $stmt->execute();
        
    } catch(PDOException $e) {
        throw new Exception("Database error: " . $e->getMessage());
    }
}

// Helper method - Check if course group exists
private function courseGroupExists($id) {
    $query = "SELECT COUNT(*) as count FROM course_groups WHERE id = :id";
    $stmt = $this->conn->prepare($query);
    $stmt->bindParam(":id", $id, PDO::PARAM_INT);
    $stmt->execute();
    $result = $stmt->fetch(PDO::FETCH_ASSOC);
    
    return $result['count'] > 0;
}
        
        // Existing method - Keep as is
        public function getCourseDetails($courseId = null) {
            try {
                $query = "SELECT 
                            c.id, 
                            c.name, 
                            c.position, 
                            c.description,
                            cg.name as course_group_name,
                            cg.id as course_group_id,
                            acs.id as academic_session_id,
                            acs.name as academic_session_name,
                            acs.start_date as session_start_date,
                            acs.end_date as session_end_date,
                            acs.is_default as session_is_default
                        FROM courses c
                        LEFT JOIN course_groups cg ON c.course_group_id = cg.id
                        LEFT JOIN academic_sessions acs ON c.academic_session_id = acs.id";
                
                if($courseId) {
                    $query .= " WHERE c.id = :course_id";
                }
                
                $stmt = $this->conn->prepare($query);
                
                if($courseId) {
                    $stmt->bindParam(":course_id", $courseId, PDO::PARAM_INT);
                }
                
                $stmt->execute();
                
                $result = $stmt->fetchAll(PDO::FETCH_ASSOC);
                
                if (!$result) {
                    return array();
                }
                
                // Format dates
                foreach($result as &$row) {
                    if(isset($row['session_start_date'])) {
                        $row['session_start_date'] = date('Y-m-d', strtotime($row['session_start_date']));
                    }
                    if(isset($row['session_end_date'])) {
                        $row['session_end_date'] = date('Y-m-d', strtotime($row['session_end_date']));
                    }
                }
                
                return $result;
            } catch(PDOException $e) {
                throw new Exception("Error fetching course details: " . $e->getMessage());
            }
        }
        
        // Existing method - Keep as is
        public function getBatchDetails($batchId = null) {
            try {
                $query = "SELECT 
                            b.id, 
                            b.name, 
                            b.position, 
                            b.description,
                            c.id as course_id,
                            c.name as course_name,
                            eg.id as exam_grade_id,
                            eg.name as exam_grade_name,
                            eo.id as exam_observation_id,
                            eo.name as exam_observation_name,
                            acs.id as academic_session_id,
                            acs.name as academic_session_name,
                            acs.start_date as session_start_date,
                            acs.end_date as session_end_date
                        FROM batches b
                        LEFT JOIN courses c ON b.course_id = c.id
                        LEFT JOIN exam_grades eg ON b.exam_grade_id = eg.id
                        LEFT JOIN exam_observations eo ON b.exam_observation_id = eo.id
                        LEFT JOIN academic_sessions acs ON c.academic_session_id = acs.id";
                
                if($batchId) {
                    $query .= " WHERE b.id = :batch_id";
                }
                
                $stmt = $this->conn->prepare($query);
                
                if($batchId) {
                    $stmt->bindParam(":batch_id", $batchId, PDO::PARAM_INT);
                }
                
                $stmt->execute();
                
                $result = $stmt->fetchAll(PDO::FETCH_ASSOC);
                
                if (!$result) {
                    return array();
                }
                
                // Format dates
                foreach($result as &$row) {
                    if(isset($row['session_start_date'])) {
                        $row['session_start_date'] = date('Y-m-d', strtotime($row['session_start_date']));
                    }
                    if(isset($row['session_end_date'])) {
                        $row['session_end_date'] = date('Y-m-d', strtotime($row['session_end_date']));
                    }
                }
                
                return $result;
            } catch(PDOException $e) {
                throw new Exception("Error fetching batch details: " . $e->getMessage());
            }
        }
        
        // New method - Create a course
        public function createCourse($data) {
            try {
                // Validate required fields
                if(!isset($data->name) || !isset($data->academic_session_id)) {
                    throw new Exception("Name and academic session ID are required fields");
                }
                
                $query = "INSERT INTO courses (
                            name, 
                            description, 
                            position,
                            academic_session_id,
                            course_group_id,
                            created_at, 
                            updated_at
                          ) VALUES (
                            :name, 
                            :description, 
                            :position,
                            :academic_session_id,
                            :course_group_id,
                            NOW(), 
                            NOW()
                          )";
                
                $stmt = $this->conn->prepare($query);
                
                // Clean and bind data
                $name = htmlspecialchars(strip_tags($data->name));
                $description = isset($data->description) ? htmlspecialchars(strip_tags($data->description)) : null;
                $position = isset($data->position) ? intval($data->position) : 0;
                $academicSessionId = intval($data->academic_session_id);
                $courseGroupId = isset($data->course_group_id) ? intval($data->course_group_id) : null;
                
                $stmt->bindParam(":name", $name);
                $stmt->bindParam(":description", $description);
                $stmt->bindParam(":position", $position);
                $stmt->bindParam(":academic_session_id", $academicSessionId);
                $stmt->bindParam(":course_group_id", $courseGroupId);
                
                if($stmt->execute()) {
                    $courseId = $this->conn->lastInsertId();
                    return array(
                        "id" => $courseId,
                        "name" => $name,
                        "description" => $description,
                        "position" => $position,
                        "academic_session_id" => $academicSessionId,
                        "course_group_id" => $courseGroupId
                    );
                }
                
                throw new Exception("Unknown error occurred when creating course");
                
            } catch(PDOException $e) {
                throw new Exception("Database error: " . $e->getMessage());
            }
        }
        
        // New method - Update a course
        public function updateCourse($id, $data) {
            try {
                // First check if course exists
                $courseExists = $this->courseExists($id);
                if(!$courseExists) {
                    throw new Exception("Course with ID $id not found");
                }
                
                // Build query based on provided fields
                $updateFields = array();
                $params = array();
                
                if(isset($data->name)) {
                    $updateFields[] = "name = :name";
                    $params[':name'] = htmlspecialchars(strip_tags($data->name));
                }
                
                if(isset($data->description)) {
                    $updateFields[] = "description = :description";
                    $params[':description'] = htmlspecialchars(strip_tags($data->description));
                }
                
                if(isset($data->position)) {
                    $updateFields[] = "position = :position";
                    $params[':position'] = intval($data->position);
                }
                
                if(isset($data->academic_session_id)) {
                    $updateFields[] = "academic_session_id = :academic_session_id";
                    $params[':academic_session_id'] = intval($data->academic_session_id);
                }
                
                if(isset($data->course_group_id)) {
                    $updateFields[] = "course_group_id = :course_group_id";
                    $params[':course_group_id'] = intval($data->course_group_id);
                }
                
                // Add updated_at timestamp
                $updateFields[] = "updated_at = NOW()";
                
                // If no fields to update, return early
                if(empty($updateFields)) {
                    throw new Exception("No fields provided for update");
                }
                
                $query = "UPDATE courses SET " . implode(", ", $updateFields) . " WHERE id = :id";
                $params[':id'] = $id;
                
                $stmt = $this->conn->prepare($query);
                
                foreach($params as $param => $value) {
                    $stmt->bindValue($param, $value);
                }
                
                if($stmt->execute()) {
                    // Return updated course data
                    $updatedCourse = $this->getCourseDetails($id);
                    return !empty($updatedCourse) ? $updatedCourse[0] : array();
                }
                
                throw new Exception("Unknown error occurred when updating course");
                
            } catch(PDOException $e) {
                throw new Exception("Database error: " . $e->getMessage());
            }
        }
        
        // New method - Delete a course
        public function deleteCourse($id) {
            try {
                // First check if course exists
                $courseExists = $this->courseExists($id);
                if(!$courseExists) {
                    throw new Exception("Course with ID $id not found");
                }
                
                // Check if course has associated batches
                $query = "SELECT COUNT(*) as batch_count FROM batches WHERE course_id = :id";
                $stmt = $this->conn->prepare($query);
                $stmt->bindParam(":id", $id, PDO::PARAM_INT);
                $stmt->execute();
                $result = $stmt->fetch(PDO::FETCH_ASSOC);
                
                if($result['batch_count'] > 0) {
                    throw new Exception("Cannot delete course with ID $id because it has associated batches");
                }
                
                // Delete the course
                $query = "DELETE FROM courses WHERE id = :id";
                $stmt = $this->conn->prepare($query);
                $stmt->bindParam(":id", $id, PDO::PARAM_INT);
                
                return $stmt->execute();
                
            } catch(PDOException $e) {
                throw new Exception("Database error: " . $e->getMessage());
            }
        }
        
        // New method - Create a batch
        public function createBatch($data) {
            try {
                // Validate required fields
                if(!isset($data->name) || !isset($data->course_id)) {
                    throw new Exception("Name and course ID are required fields");
                }
                
                // Check if course exists
                $courseExists = $this->courseExists($data->course_id);
                if(!$courseExists) {
                    throw new Exception("Course with ID {$data->course_id} not found");
                }
                
                $query = "INSERT INTO batches (
                            name, 
                            description, 
                            position,
                            course_id,
                            exam_grade_id,
                            exam_observation_id,
                            created_at, 
                            updated_at
                          ) VALUES (
                            :name, 
                            :description, 
                            :position,
                            :course_id,
                            :exam_grade_id,
                            :exam_observation_id,
                            NOW(), 
                            NOW()
                          )";
                
                $stmt = $this->conn->prepare($query);
                
                // Clean and bind data
                $name = htmlspecialchars(strip_tags($data->name));
                $description = isset($data->description) ? htmlspecialchars(strip_tags($data->description)) : null;
                $position = isset($data->position) ? intval($data->position) : 0;
                $courseId = intval($data->course_id);
                $examGradeId = isset($data->exam_grade_id) ? intval($data->exam_grade_id) : null;
                $examObservationId = isset($data->exam_observation_id) ? intval($data->exam_observation_id) : null;
                
                $stmt->bindParam(":name", $name);
                $stmt->bindParam(":description", $description);
                $stmt->bindParam(":position", $position);
                $stmt->bindParam(":course_id", $courseId);
                $stmt->bindParam(":exam_grade_id", $examGradeId);
                $stmt->bindParam(":exam_observation_id", $examObservationId);
                
                if($stmt->execute()) {
                    $batchId = $this->conn->lastInsertId();
                    return array(
                        "id" => $batchId,
                        "name" => $name,
                        "description" => $description,
                        "position" => $position,
                        "course_id" => $courseId,
                        "exam_grade_id" => $examGradeId,
                        "exam_observation_id" => $examObservationId
                    );
                }
                
                throw new Exception("Unknown error occurred when creating batch");
                
            } catch(PDOException $e) {
                throw new Exception("Database error: " . $e->getMessage());
            }
        }
        
        // New method - Update a batch
        public function updateBatch($id, $data) {
            try {
                // First check if batch exists
                $batchExists = $this->batchExists($id);
                if(!$batchExists) {
                    throw new Exception("Batch with ID $id not found");
                }
                
                // Build query based on provided fields
                $updateFields = array();
                $params = array();
                
                if(isset($data->name)) {
                    $updateFields[] = "name = :name";
                    $params[':name'] = htmlspecialchars(strip_tags($data->name));
                }
                
                if(isset($data->description)) {
                    $updateFields[] = "description = :description";
                    $params[':description'] = htmlspecialchars(strip_tags($data->description));
                }
                
                if(isset($data->position)) {
                    $updateFields[] = "position = :position";
                    $params[':position'] = intval($data->position);
                }
                
                if(isset($data->course_id)) {
                    // Check if course exists
                    $courseExists = $this->courseExists($data->course_id);
                    if(!$courseExists) {
                        throw new Exception("Course with ID {$data->course_id} not found");
                    }
                    
                    $updateFields[] = "course_id = :course_id";
                    $params[':course_id'] = intval($data->course_id);
                }
                
                if(isset($data->exam_grade_id)) {
                    $updateFields[] = "exam_grade_id = :exam_grade_id";
                    $params[':exam_grade_id'] = intval($data->exam_grade_id);
                }
                
                if(isset($data->exam_observation_id)) {
                    $updateFields[] = "exam_observation_id = :exam_observation_id";
                    $params[':exam_observation_id'] = intval($data->exam_observation_id);
                }
                
                // Add updated_at timestamp
                $updateFields[] = "updated_at = NOW()";
                
                // If no fields to update, return early
                if(empty($updateFields)) {
                    throw new Exception("No fields provided for update");
                }
                
                $query = "UPDATE batches SET " . implode(", ", $updateFields) . " WHERE id = :id";
                $params[':id'] = $id;
                
                $stmt = $this->conn->prepare($query);
                
                foreach($params as $param => $value) {
                    $stmt->bindValue($param, $value);
                }
                
                if($stmt->execute()) {
                    // Return updated batch data
                    $updatedBatch = $this->getBatchDetails($id);
                    return !empty($updatedBatch) ? $updatedBatch[0] : array();
                }
                
                throw new Exception("Unknown error occurred when updating batch");
                
            } catch(PDOException $e) {
                throw new Exception("Database error: " . $e->getMessage());
            }
        }
        
        // New method - Delete a batch
        public function deleteBatch($id) {
            try {
                // First check if batch exists
                $batchExists = $this->batchExists($id);
                if(!$batchExists) {
                    throw new Exception("Batch with ID $id not found");
                }
                
                // Check if batch has associated records
                // This check can be expanded based on your database relationships
                // Example: check student_records, timetables, etc.
                $query = "SELECT COUNT(*) as record_count FROM student_records WHERE batch_id = :id";
                $stmt = $this->conn->prepare($query);
                $stmt->bindParam(":id", $id, PDO::PARAM_INT);
                $stmt->execute();
                $result = $stmt->fetch(PDO::FETCH_ASSOC);
                
                if($result['record_count'] > 0) {
                    throw new Exception("Cannot delete batch with ID $id because it has associated student records");
                }
                
                // Delete the batch
                $query = "DELETE FROM batches WHERE id = :id";
                $stmt = $this->conn->prepare($query);
                $stmt->bindParam(":id", $id, PDO::PARAM_INT);
                
                return $stmt->execute();
                
            } catch(PDOException $e) {
                throw new Exception("Database error: " . $e->getMessage());
            }
        }
        
        // Helper method - Check if course exists
        private function courseExists($id) {
            $query = "SELECT COUNT(*) as count FROM courses WHERE id = :id";
            $stmt = $this->conn->prepare($query);
            $stmt->bindParam(":id", $id, PDO::PARAM_INT);
            $stmt->execute();
            $result = $stmt->fetch(PDO::FETCH_ASSOC);
            
            return $result['count'] > 0;
        }
        
        // Helper method - Check if batch exists
        private function batchExists($id) {
            $query = "SELECT COUNT(*) as count FROM batches WHERE id = :id";
            $stmt = $this->conn->prepare($query);
            $stmt->bindParam(":id", $id, PDO::PARAM_INT);
            $stmt->execute();
            $result = $stmt->fetch(PDO::FETCH_ASSOC);
            
            return $result['count'] > 0;
        }
    }

    // Initialize CourseAndBatch object
    $courseAndBatch = new CourseAndBatch($db);

    // Handle different HTTP methods
    $method = $_SERVER['REQUEST_METHOD'];
    
    switch($method) {
       // Add this to the GET section of the switch statement

case 'GET':
    // Get parameters
    $courseId = isset($_GET['course_id']) ? intval($_GET['course_id']) : null;
    $batchId = isset($_GET['batch_id']) ? intval($_GET['batch_id']) : null;
    $courseGroupId = isset($_GET['course_group_id']) ? intval($_GET['course_group_id']) : null;

    // Get data based on parameters
    $response = array();

    // Handle course group request
    if ($courseGroupId) {
        $courseGroupData = $courseAndBatch->getCourseGroupDetails($courseGroupId);
        if(empty($courseGroupData)) {
            $response['course_group'] = array();
            $response['group_courses'] = array();
        } else {
            $response['course_group'] = $courseGroupData;
            
            // Get associated courses for this course group
            $courses = $courseAndBatch->getCourseDetails();
            $groupCourses = array_filter($courses, function($course) use ($courseGroupId) {
                return isset($course['course_group_id']) && $course['course_group_id'] == $courseGroupId;
            });
            $response['group_courses'] = array_values($groupCourses);
        }
        
        http_response_code(200);
        echo json_encode(array(
            "status" => true,
            "message" => "Course group data retrieved successfully",
            "data" => $response
        ));
        exit();
    } 
    // Handle all course groups request
    else if (isset($_GET['list_course_groups'])) {
        $response['course_groups'] = $courseAndBatch->getCourseGroupDetails();
        
        http_response_code(200);
        echo json_encode(array(
            "status" => true,
            "message" => "All course groups retrieved successfully",
            "data" => $response
        ));
        exit();
    }
    // Regular course/batch requests (existing code)
    else if($courseId || $batchId) {
        if($courseId) {
            $courseData = $courseAndBatch->getCourseDetails($courseId);
            if(empty($courseData)) {
                $response['course'] = array();
                $response['course_batches'] = array();
            } else {
                $response['course'] = $courseData;
                
                // Get associated batches for this course
                $batches = $courseAndBatch->getBatchDetails();
                $courseBatches = array_filter($batches, function($batch) use ($courseId) {
                    return $batch['course_id'] == $courseId;
                });
                $response['course_batches'] = array_values($courseBatches);
            }
        }
        
        if($batchId) {
            $batchData = $courseAndBatch->getBatchDetails($batchId);
            $response['batch'] = empty($batchData) ? array() : $batchData;
        }
        
        http_response_code(200);
        echo json_encode(array(
            "status" => true,
            "message" => "Data retrieved successfully",
            "data" => $response
        ));
    } else {
        // Return all data if no specific ID is provided
        $response['courses'] = $courseAndBatch->getCourseDetails();
        $response['batches'] = $courseAndBatch->getBatchDetails();
        $response['course_groups'] = $courseAndBatch->getCourseGroupDetails();
        
        http_response_code(200);
        echo json_encode(array(
            "status" => true,
            "message" => "All data retrieved successfully",
            "data" => $response
        ));
    }
    break;
            
            
        case 'POST':
            
            // Create new course or batch
            $data = json_decode(file_get_contents("php://input"));
            
            if(!$data) {
                http_response_code(400);
                echo json_encode(array(
                    "status" => false,
                    "message" => "Invalid input data"
                ));
                exit();
            }
            
            if(isset($data->entity_type)) {
                if($data->entity_type === 'course') {
                    // Create course
                    $result = $courseAndBatch->createCourse($data);
                    http_response_code(201);
                    echo json_encode(array(
                        "status" => true,
                        "message" => "Course created successfully",
                        "data" => $result
                    ));
                } else if($data->entity_type === 'batch') {
                    // Create batch
                    $result = $courseAndBatch->createBatch($data);
                    http_response_code(201);
                    echo json_encode(array(
                        "status" => true,
                        "message" => "Batch created successfully",
                        "data" => $result
                    ));
                } else if($data->entity_type === 'course_group') {
        // Create course group (new code)
        $result = $courseAndBatch->createCourseGroup($data);
        http_response_code(201);
        echo json_encode(array(
            "status" => true,
            "message" => "Course group created successfully",
            "data" => $result
        ));
    } else {
        http_response_code(400);
        echo json_encode(array(
            "status" => false,
            "message" => "Invalid entity type. Must be 'course', 'batch', or 'course_group'"
        ));
    }
} else {
    http_response_code(400);
    echo json_encode(array(
        "status" => false,
        "message" => "Entity type not specified. Must be 'course', 'batch', or 'course_group'"
    ));
}

            break;
            
            
        case 'PUT':
            // Update existing course or batch
            $data = json_decode(file_get_contents("php://input"));
            
            if(!$data) {
                http_response_code(400);
                echo json_encode(array(
                    "status" => false,
                    "message" => "Invalid input data"
                ));
                exit();
            }
            
            if(isset($data->entity_type) && isset($data->id)) {
                $id = intval($data->id);
                
                if($data->entity_type === 'course') {
                    // Update course
                    $result = $courseAndBatch->updateCourse($id, $data);
                    http_response_code(200);
                    echo json_encode(array(
                        "status" => true,
                        "message" => "Course updated successfully",
                        "data" => $result
                    ));
                } else if($data->entity_type === 'batch') {
                    // Update batch
                    $result = $courseAndBatch->updateBatch($id, $data);
                    http_response_code(200);
                    echo json_encode(array(
                        "status" => true,
                        "message" => "Batch updated successfully",
                        "data" => $result
                    ));
                } else if($data->entity_type === 'course_group') {
        // Update course group (new code)
        $result = $courseAndBatch->updateCourseGroup($id, $data);
        http_response_code(200);
        echo json_encode(array(
            "status" => true,
            "message" => "Course group updated successfully",
            "data" => $result
        ));
    } else {
        http_response_code(400);
        echo json_encode(array(
            "status" => false,
            "message" => "Invalid entity type. Must be 'course', 'batch', or 'course_group'"
        ));
    }
} else {
    http_response_code(400);
    echo json_encode(array(
        "status" => false,
        "message" => "Entity type and ID are required"
    ));
}
            break;
            
        case 'DELETE':
            // Delete course or batch
            $data = json_decode(file_get_contents("php://input"));
            
            if(!$data) {
                http_response_code(400);
                echo json_encode(array(
                    "status" => false,
                    "message" => "Invalid input data"
                ));
                exit();
            }
            
            if(isset($data->entity_type) && isset($data->id)) {
                $id = intval($data->id);
                
                if($data->entity_type === 'course') {
                    // Delete course
                    $result = $courseAndBatch->deleteCourse($id);
                    http_response_code(200);
                    echo json_encode(array(
                        "status" => true,
                        "message" => "Course deleted successfully"
                    ));
                } else if($data->entity_type === 'batch') {
                    // Delete batch
                    $result = $courseAndBatch->deleteBatch($id);
                    http_response_code(200);
                    echo json_encode(array(
                        "status" => true,
                        "message" => "Batch deleted successfully"
                    ));
                } else if($data->entity_type === 'course_group') {
        // Delete course group (new code)
        $result = $courseAndBatch->deleteCourseGroup($id);
        http_response_code(200);
        echo json_encode(array(
            "status" => true,
            "message" => "Course group deleted successfully"
        ));
    } else {
        http_response_code(400);
        echo json_encode(array(
            "status" => false,
            "message" => "Invalid entity type. Must be 'course', 'batch', or 'course_group'"
        ));
    }
} else {
    http_response_code(400);
    echo json_encode(array(
        "status" => false,
        "message" => "Entity type and ID are required"
    ));
}
            break;
            
        default:
            http_response_code(405);
            echo json_encode(array(
                "status" => false,
                "message" => "Method not allowed"
            ));
            break;
    }

} catch(Exception $e) {
    http_response_code(500);
    echo json_encode(array(
        "status" => false,
        "message" => "Server error occurred",
        "error" => $e->getMessage(),
        "trace" => debug_backtrace()
    ));
}
?>