<?php
// Headers
header("Access-Control-Allow-Origin: *");
header("Content-Type: application/json; charset=UTF-8");
header("Access-Control-Allow-Methods: POST, GET, OPTIONS");
header("Access-Control-Max-Age: 3600");
header("Access-Control-Allow-Headers: Content-Type, Access-Control-Allow-Headers, Authorization, X-Requested-With");

// Handle preflight requests
if ($_SERVER['REQUEST_METHOD'] == 'OPTIONS') {
    http_response_code(200);
    exit();
}

// Include dependencies
require_once '../config/Database.php';
require_once '../middleware/Auth.php';

// Initialize API handlers
require_once '../models/AcademicSession.php';
require_once '../models/CourseGroup.php';
require_once '../models/Course.php';
require_once '../models/Batch.php';
require_once '../models/Subject.php';

try {
    // Initialize database
    $database = new Database();
    $db = $database->getConnection();
    
    // Validate token
    $auth = new Auth($db);
    
    // Get token from headers
    $allHeaders = getallheaders();
    $token = null;
    
    // Check Authorization header
    foreach ($allHeaders as $name => $value) {
        if (strtolower($name) === 'authorization') {
            if (preg_match('/Bearer\s+(.+)/', $value, $matches)) {
                $token = $matches[1];
            } else {
                $token = $value;
            }
            break;
        }
    }
    
    // Check HTTP_AUTHORIZATION if not found in headers
    if (!$token && isset($_SERVER['HTTP_AUTHORIZATION'])) {
        $authHeader = $_SERVER['HTTP_AUTHORIZATION'];
        if (preg_match('/Bearer\s+(.+)/', $authHeader, $matches)) {
            $token = $matches[1];
        } else {
            $token = $authHeader;
        }
    }
    
    // Last resort: check GET parameter
    if (!$token && isset($_GET['token'])) {
        $token = $_GET['token'];
    }
    
    // Validate token
    if (!$token || !$auth->tokenExists($token)) {
        http_response_code(401);
        echo json_encode([
            "status" => false,
            "message" => "Unauthorized: Invalid or missing token"
        ]);
        exit();
    }
    
    $user = $auth->validateToken($token);
    if (!$user) {
        http_response_code(401);
        echo json_encode([
            "status" => false,
            "message" => "Unauthorized: Token validation failed"
        ]);
        exit();
    }
    
    // Process request
    $data = json_decode(file_get_contents("php://input"), true);
    
    // For GET requests, get parameters from $_GET
    if ($_SERVER['REQUEST_METHOD'] === 'GET') {
        $action = isset($_GET['action']) ? $_GET['action'] : 'list';
        $entity = isset($_GET['entity']) ? $_GET['entity'] : null;
        $id = isset($_GET['id']) ? intval($_GET['id']) : null;
    } else {
        // For POST requests, get from JSON body
        $action = isset($data['action']) ? $data['action'] : null;
        $entity = isset($data['entity']) ? $data['entity'] : null;
        $id = isset($data['id']) ? intval($data['id']) : null;
    }
    
    // Validate required parameters
    if (!$action || !$entity) {
        http_response_code(400);
        echo json_encode([
            "status" => false,
            "message" => "Missing required parameters: action and entity are required"
        ]);
        exit();
    }
    
    // Initialize the appropriate handler based on entity
    switch ($entity) {
        case 'academic_session':
            $handler = new AcademicSession($db);
            break;
        case 'course_group':
            $handler = new CourseGroup($db);
            break;
        case 'course':
            $handler = new Course($db);
            break;
        case 'batch':
            $handler = new Batch($db);
            break;
        case 'subject':
            $handler = new Subject($db);
            break;
        default:
            http_response_code(400);
            echo json_encode([
                "status" => false,
                "message" => "Invalid entity type: " . $entity
            ]);
            exit();
    }
    
    // Process the request based on action
    switch ($action) {
        case 'list':
            $result = $handler->getAll();
            break;
            
        case 'get':
            if (!$id) {
                http_response_code(400);
                echo json_encode([
                    "status" => false,
                    "message" => "ID is required for 'get' action"
                ]);
                exit();
            }
            $result = $handler->getById($id);
            break;
            
        case 'create':
            if ($_SERVER['REQUEST_METHOD'] === 'GET') {
                http_response_code(405);
                echo json_encode([
                    "status" => false,
                    "message" => "Method Not Allowed: Create requires POST"
                ]);
                exit();
            }
            
            $entityData = isset($data['data']) ? $data['data'] : null;
            if (!$entityData) {
                http_response_code(400);
                echo json_encode([
                    "status" => false,
                    "message" => "Missing data for create operation"
                ]);
                exit();
            }
            
            $result = $handler->create($entityData);
            break;
            
        case 'update':
            if ($_SERVER['REQUEST_METHOD'] === 'GET') {
                http_response_code(405);
                echo json_encode([
                    "status" => false,
                    "message" => "Method Not Allowed: Update requires POST"
                ]);
                exit();
            }
            
            if (!$id) {
                http_response_code(400);
                echo json_encode([
                    "status" => false,
                    "message" => "ID is required for update operation"
                ]);
                exit();
            }
            
            $entityData = isset($data['data']) ? $data['data'] : null;
            if (!$entityData) {
                http_response_code(400);
                echo json_encode([
                    "status" => false,
                    "message" => "Missing data for update operation"
                ]);
                exit();
            }
            
            $result = $handler->update($id, $entityData);
            break;
            
        case 'delete':
            if ($_SERVER['REQUEST_METHOD'] === 'GET') {
                http_response_code(405);
                echo json_encode([
                    "status" => false,
                    "message" => "Method Not Allowed: Delete requires POST"
                ]);
                exit();
            }
            
            if (!$id) {
                http_response_code(400);
                echo json_encode([
                    "status" => false,
                    "message" => "ID is required for delete operation"
                ]);
                exit();
            }
            
            $result = $handler->delete($id);
            break;
            
        default:
            http_response_code(400);
            echo json_encode([
                "status" => false,
                "message" => "Invalid action: " . $action
            ]);
            exit();
    }
    
    // Send response
    http_response_code(200);
    echo json_encode([
        "status" => true,
        "message" => "Operation completed successfully",
        "data" => $result
    ]);
    
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode([
        "status" => false,
        "message" => "Server error: " . $e->getMessage()
    ]);
}
?>