// Global JavaScript Functions

// Register Service Worker for PWA
if ('serviceWorker' in navigator) {
    window.addEventListener('load', function() {
        navigator.serviceWorker.register('/service-worker.js')
            .then(function(registration) {
                console.log('ServiceWorker registration successful with scope: ', registration.scope);
            })
            .catch(function(error) {
                console.log('ServiceWorker registration failed: ', error);
            });
    });
}

// Utility Functions
const Utils = {
    // Format date
    formatDate: function(dateString) {
        const options = { year: 'numeric', month: 'long', day: 'numeric' };
        return new Date(dateString).toLocaleDateString(undefined, options);
    },
    
    // Format currency
    formatCurrency: function(amount) {
        return new Intl.NumberFormat('en-IN', {
            style: 'currency',
            currency: 'INR',
            minimumFractionDigits: 0,
            maximumFractionDigits: 0
        }).format(amount);
    },
    
    // Format number with commas
    formatNumber: function(num) {
        return new Intl.NumberFormat('en-IN').format(num);
    },
    
    // Get initials from name
    getInitials: function(name) {
        return name
            .split(' ')
            .map(word => word.charAt(0))
            .join('')
            .toUpperCase();
    },
    
    // Truncate text
    truncateText: function(text, length = 30) {
        if (text.length <= length) return text;
        return text.substring(0, length) + '...';
    }
};

// Auth Functions
const Auth = {
    // Get token
    getToken: function() {
        return localStorage.getItem('auth_token');
    },
    
    // Get user data
    getUserData: function() {
        const userData = localStorage.getItem('user_data');
        return userData ? JSON.parse(userData) : null;
    },
    
    // Check if user is authenticated
    isAuthenticated: function() {
        return !!this.getToken();
    },
    
    // Logout
    logout: function() {
        localStorage.removeItem('auth_token');
        localStorage.removeItem('user_data');
        window.location.href = 'index.html';
    },
    
    // Check role
    hasRole: function(role) {
        const userData = this.getUserData();
        if (!userData || !userData.roles) return false;
        return userData.roles.includes(role);
    }
};

// API Functions
const API = {
    // Base URL
    baseURL: 'https://edmapi.ed5.in/api',
    
    // Headers
    getHeaders: function() {
        const token = Auth.getToken();
        return {
            'Content-Type': 'application/json',
            'Authorization': token ? token : ''
        };
    },
    
    // GET request
    async get(endpoint) {
        try {
            const response = await fetch(`${this.baseURL}/${endpoint}`, {
                method: 'GET',
                headers: this.getHeaders()
            });
            
            return await this.handleResponse(response);
        } catch (error) {
            console.error('API GET error:', error);
            throw error;
        }
    },
    
    // POST request
    async post(endpoint, data) {
        try {
            const response = await fetch(`${this.baseURL}/${endpoint}`, {
                method: 'POST',
                headers: this.getHeaders(),
                body: JSON.stringify(data)
            });
            
            return await this.handleResponse(response);
        } catch (error) {
            console.error('API POST error:', error);
            throw error;
        }
    },
    
    // PUT request
    async put(endpoint, data) {
        try {
            const response = await fetch(`${this.baseURL}/${endpoint}`, {
                method: 'PUT',
                headers: this.getHeaders(),
                body: JSON.stringify(data)
            });
            
            return await this.handleResponse(response);
        } catch (error) {
            console.error('API PUT error:', error);
            throw error;
        }
    },
    
    // DELETE request
    async delete(endpoint) {
        try {
            const response = await fetch(`${this.baseURL}/${endpoint}`, {
                method: 'DELETE',
                headers: this.getHeaders()
            });
            
            return await this.handleResponse(response);
        } catch (error) {
            console.error('API DELETE error:', error);
            throw error;
        }
    },
    
    // Handle response
    async handleResponse(response) {
        const data = await response.json();
        
        if (!response.ok) {
            if (response.status === 401) {
                // Unauthorized - redirect to login
                Auth.logout();
            }
            
            throw new Error(data.message || 'Something went wrong');
        }
        
        return data;
    }
};