<?php
// Enable error reporting for debugging
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Headers
header("Access-Control-Allow-Origin: *");
header("Content-Type: application/json; charset=UTF-8");
header("Access-Control-Allow-Methods: GET, OPTIONS");
header("Access-Control-Max-Age: 3600");
header("Access-Control-Allow-Headers: Content-Type, Access-Control-Allow-Headers, Authorization, X-Requested-With");

// Handle preflight requests
if ($_SERVER['REQUEST_METHOD'] == 'OPTIONS') {
    http_response_code(200);
    exit();
}

// Debug: Log the raw request
error_log("Raw request received: " . file_get_contents('php://input'));

try {
    // Include database and auth class
    require_once '../config/Database.php';
    require_once '../middleware/Auth.php';

    // Initialize database
    $database = new Database();
    $db = $database->getConnection();

    // Check database connection
    if (!$db) {
        throw new Exception("Database connection failed");
    }

    // Initialize Auth
    $auth = new Auth($db);

    // Get the token from multiple possible sources
    $token = null;
    
    // 1. Try from getallheaders() first (case-insensitive)
    $allHeaders = getallheaders();
    error_log("All headers: " . print_r($allHeaders, true));
    
    foreach ($allHeaders as $name => $value) {
        if (strtolower($name) === 'authorization') {
            // Extract token - handle both "Bearer token" and raw token formats
            if (preg_match('/Bearer\s+(.+)/', $value, $matches)) {
                $token = $matches[1];
            } else {
                $token = $value;
            }
            error_log("Found token in getallheaders(): " . substr($token, 0, 10) . "...");
            break;
        }
    }
    
    // 2. Try from $_SERVER if still not found
    if (!$token) {
        // Apache specific
        if (isset($_SERVER['HTTP_AUTHORIZATION'])) {
            $authHeader = $_SERVER['HTTP_AUTHORIZATION'];
            
            if (preg_match('/Bearer\s+(.+)/', $authHeader, $matches)) {
                $token = $matches[1];
            } else {
                $token = $authHeader;
            }
            error_log("Found token in \$_SERVER['HTTP_AUTHORIZATION']: " . substr($token, 0, 10) . "...");
        }
        // Apache with mod_rewrite
        elseif (isset($_SERVER['REDIRECT_HTTP_AUTHORIZATION'])) {
            $authHeader = $_SERVER['REDIRECT_HTTP_AUTHORIZATION'];
            
            if (preg_match('/Bearer\s+(.+)/', $authHeader, $matches)) {
                $token = $matches[1];
            } else {
                $token = $authHeader;
            }
            error_log("Found token in \$_SERVER['REDIRECT_HTTP_AUTHORIZATION']: " . substr($token, 0, 10) . "...");
        }
        // Try to get raw headers
        else {
            foreach ($_SERVER as $key => $value) {
                if (substr($key, 0, 5) === 'HTTP_') {
                    $key = str_replace(' ', '-', ucwords(strtolower(str_replace('_', ' ', substr($key, 5)))));
                    if (strtolower($key) === 'authorization') {
                        if (preg_match('/Bearer\s+(.+)/', $value, $matches)) {
                            $token = $matches[1];
                        } else {
                            $token = $value;
                        }
                        error_log("Found token in \$_SERVER HTTP_ variable: " . substr($token, 0, 10) . "...");
                        break;
                    }
                }
            }
        }
    }
    
    // 3. Last resort: Look in GET parameters
    if (!$token && isset($_GET['token'])) {
        $token = $_GET['token'];
        error_log("Found token in GET parameter: " . substr($token, 0, 10) . "...");
    }

    // Validate token
    if (!$token) {
        http_response_code(401);
        echo json_encode(array(
            "status" => false,
            "message" => "No authorization token provided",
            "headers_received" => $allHeaders
        ));
        exit();
    }

    // Check if token exists in database before validation
    if (!$auth->tokenExists($token)) {
        error_log("Token does not exist in database: " . substr($token, 0, 10) . "...");
        http_response_code(401);
        echo json_encode(array(
            "status" => false,
            "message" => "Invalid token: not found in database"
        ));
        exit();
    }

    $user = $auth->validateToken($token);
    
    if (!$user) {
        http_response_code(401);
        echo json_encode(array(
            "status" => false,
            "message" => "Invalid or expired token"
        ));
        exit();
    }

    // Dashboard data class
    class DashboardSummary {
        private $conn;
        private $today;
        private $currentMonth;
        private $currentYear;
        private $academicSessionId;
        
        public function __construct($db) {
            $this->conn = $db;
            $this->today = date('Y-m-d');
            $this->currentMonth = date('m');
            $this->currentYear = date('Y');
            $this->academicSessionId = $this->getCurrentAcademicSession();
        }
        
        private function getCurrentAcademicSession() {
            try {
                $query = "SELECT id FROM academic_sessions WHERE is_default = 1 LIMIT 1";
                $stmt = $this->conn->prepare($query);
                $stmt->execute();
                
                if ($stmt->rowCount() > 0) {
                    $row = $stmt->fetch(PDO::FETCH_ASSOC);
                    return $row['id'];
                } else {
                    // If no default session, get the most recent one
                    $query = "SELECT id FROM academic_sessions ORDER BY end_date DESC LIMIT 1";
                    $stmt = $this->conn->prepare($query);
                    $stmt->execute();
                    
                    if ($stmt->rowCount() > 0) {
                        $row = $stmt->fetch(PDO::FETCH_ASSOC);
                        return $row['id'];
                    }
                }
                return null;
            } catch(PDOException $e) {
                throw new Exception("Error fetching current academic session: " . $e->getMessage());
            }
        }
        
        // Get attendance summary
// Get attendance summary
public function getAttendanceSummary() {
    try {
        // Student attendance - today's summary
        $studentAttendance = [
            'total' => 0,
            'present' => 0,
            'absent' => 0,
            'late' => 0,
            'half_day' => 0,
            'percentage' => 0,
            'batches' => [
                'total' => 0,
                'completed' => 0,
                'pending' => 0,
                'completion_percentage' => 0,
                'details' => []
            ]
        ];
        
        // Count total students in current academic session
        $query = "SELECT COUNT(id) as total FROM student_records WHERE academic_session_id = :academic_session_id AND date_of_exit IS NULL";
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':academic_session_id', $this->academicSessionId);
        $stmt->execute();
        
        if ($stmt->rowCount() > 0) {
            $row = $stmt->fetch(PDO::FETCH_ASSOC);
            $studentAttendance['total'] = $row['total'];
        }
        
        // Get all active batches with course information
        $query = "SELECT b.id, b.name, c.id as course_id, c.name as course_name, COUNT(sr.id) as student_count 
                 FROM batches b 
                 JOIN courses c ON b.course_id = c.id
                 JOIN student_records sr ON b.id = sr.batch_id 
                 WHERE sr.academic_session_id = :academic_session_id 
                 AND sr.date_of_exit IS NULL 
                 GROUP BY b.id, b.name, c.id, c.name";
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':academic_session_id', $this->academicSessionId);
        $stmt->execute();
        
        $all_batches = [];
        $completed_batches = [];
        
        if ($stmt->rowCount() > 0) {
            while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
                $all_batches[$row['id']] = [
                    'id' => $row['id'],
                    'name' => $row['name'],
                    'course_id' => $row['course_id'],
                    'course_name' => $row['course_name'],
                    'student_count' => $row['student_count'],
                    'attendance_taken' => false
                ];
            }
            
            $studentAttendance['batches']['total'] = count($all_batches);
        }
        
        // Get today's attendance by batch
        $query = "SELECT sa.id, sa.batch_id, sa.attendance, b.name as batch_name, 
                 c.id as course_id, c.name as course_name
                 FROM student_attendances sa 
                 JOIN batches b ON sa.batch_id = b.id 
                 JOIN courses c ON b.course_id = c.id
                 WHERE sa.date_of_attendance = :today";
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':today', $this->today);
        $stmt->execute();
        
        if ($stmt->rowCount() > 0) {
            while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
                $attendanceData = json_decode($row['attendance'], true);
                $batch_id = $row['batch_id'];
                
                // Mark this batch as having attendance taken
                if (isset($all_batches[$batch_id])) {
                    $all_batches[$batch_id]['attendance_taken'] = true;
                    $completed_batches[] = $batch_id;
                    
                    // Add to batch details
                    $batch_stats = [
                        'id' => $batch_id,
                        'name' => $row['batch_name'],
                        'course_id' => $row['course_id'],
                        'course_name' => $row['course_name'],
                        'student_count' => isset($all_batches[$batch_id]['student_count']) ? $all_batches[$batch_id]['student_count'] : 0,
                        'present' => 0,
                        'absent' => 0,
                        'late' => 0,
                        'half_day' => 0
                    ];
                    
                    if (is_array($attendanceData)) {
                        // Count absent students (data array)
                        $absent_count = 0;
                        if (isset($attendanceData['data']) && is_array($attendanceData['data'])) {
                            $absent_count = count($attendanceData['data']);
                            $studentAttendance['absent'] += $absent_count;
                            $batch_stats['absent'] = $absent_count;
                        }
                        
                        // Count late students
                        $late_count = 0;
                        if (isset($attendanceData['late']) && is_array($attendanceData['late'])) {
                            $late_count = count($attendanceData['late']);
                            $studentAttendance['late'] += $late_count;
                            $batch_stats['late'] = $late_count;
                        }
                        
                        // Count half-day students
                        $half_day_count = 0;
                        if (isset($attendanceData['half_day']) && is_array($attendanceData['half_day'])) {
                            $half_day_count = count($attendanceData['half_day']);
                            $studentAttendance['half_day'] += $half_day_count;
                            $batch_stats['half_day'] = $half_day_count;
                        }
                        
                        // Calculate present students for this batch
                        $batch_stats['present'] = $batch_stats['student_count'] - $absent_count - $late_count - $half_day_count;
                        if ($batch_stats['present'] < 0) {
                            $batch_stats['present'] = 0;
                        }
                        $studentAttendance['present'] += $batch_stats['present'];
                    }
                    
                    $studentAttendance['batches']['details'][] = $batch_stats;
                }
            }
        }
        
        // Update batch completion stats
        $studentAttendance['batches']['completed'] = count($completed_batches);
        $studentAttendance['batches']['pending'] = $studentAttendance['batches']['total'] - $studentAttendance['batches']['completed'];
        
        if ($studentAttendance['batches']['total'] > 0) {
            $completion_percentage = ($studentAttendance['batches']['completed'] / $studentAttendance['batches']['total']) * 100;
            // Ensure exactly 2 decimal places and convert back to float
            $studentAttendance['batches']['completion_percentage'] = (float)number_format($completion_percentage, 2, '.', '');
        }
        
        // Calculate overall attendance percentage
        $total_students_counted = $studentAttendance['present'] + $studentAttendance['absent'] + $studentAttendance['late'] + $studentAttendance['half_day'];
        
        // For the percentage calculation, count late as present and half_day as 0.5 present
        $present_equivalent = $studentAttendance['present'] + $studentAttendance['late'] + ($studentAttendance['half_day'] * 0.5);
        
        if ($total_students_counted > 0) {
            $percentage = ($present_equivalent / $total_students_counted) * 100;
            // Ensure exactly 2 decimal places and convert back to float
            $studentAttendance['percentage'] = (float)number_format($percentage, 2, '.', '');
        }
        
        // Employee attendance - today's summary
        $employeeAttendance = [
            'total' => 0,
            'present' => 0,
            'absent' => 0,
            'leave' => 0,
            'percentage' => 0
        ];
        
        // Count total employees
        $query = "SELECT COUNT(id) as total FROM employees WHERE id IN (SELECT employee_id FROM employee_designations WHERE employee_id IS NOT NULL)";
        $stmt = $this->conn->prepare($query);
        $stmt->execute();
        
        if ($stmt->rowCount() > 0) {
            $row = $stmt->fetch(PDO::FETCH_ASSOC);
            $employeeAttendance['total'] = $row['total'];
        }
        
        // Get today's employee attendance types
        $query = "SELECT eat.name, COUNT(ea.id) as count 
                 FROM employee_attendances ea 
                 JOIN employee_attendance_types eat ON ea.employee_attendance_type_id = eat.id 
                 WHERE ea.date_of_attendance = :today 
                 GROUP BY eat.name";
        $stmt = $this->conn->prepare($query);
        $stmt->bindParam(':today', $this->today);
        $stmt->execute();
        
        if ($stmt->rowCount() > 0) {
            while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
                if (strtolower($row['name']) == 'present') {
                    $employeeAttendance['present'] = $row['count'];
                } else if (strtolower($row['name']) == 'absent') {
                    $employeeAttendance['absent'] = $row['count'];
                } else if (strtolower($row['name']) == 'leave') {
                    $employeeAttendance['leave'] = $row['count'];
                }
            }
        }
        
        // Calculate percentage
        $total_employee_marked = $employeeAttendance['present'] + $employeeAttendance['absent'] + $employeeAttendance['leave'];
        
        if ($total_employee_marked > 0) {
            $employee_percentage = ($employeeAttendance['present'] / $total_employee_marked) * 100;
            // Ensure exactly 2 decimal places and convert back to float
            $employeeAttendance['percentage'] = (float)number_format($employee_percentage, 2, '.', '');
        }
        
        return [
            'student' => $studentAttendance,
            'employee' => $employeeAttendance
        ];
    } catch(PDOException $e) {
        throw new Exception("Error fetching attendance summary: " . $e->getMessage());
    }
}
        
        // Get fee collection summary
        public function getFeeCollectionSummary() {
            try {
                $feeSummary = [
                    'today' => [
                        'total' => 0,
                        'paid' => 0,
                        'pending' => 0,
                        'percentage' => 0
                    ],
                    'month' => [
                        'total' => 0,
                        'paid' => 0,
                        'pending' => 0,
                        'percentage' => 0
                    ],
                    'session' => [
                        'total' => 0,
                        'paid' => 0,
                        'pending' => 0,
                        'percentage' => 0
                    ],
                    'status_breakdown' => [
                        'paid' => 0,
                        'partially_paid' => 0,
                        'pending' => 0,
                        'overdue' => 0
                    ]
                ];
                
                // Session total and paid (from student_records)
                $query = "SELECT SUM(total_fee) as total, SUM(paid_fee) as paid 
                         FROM student_records 
                         WHERE academic_session_id = :academic_session_id";
                $stmt = $this->conn->prepare($query);
                $stmt->bindParam(':academic_session_id', $this->academicSessionId);
                $stmt->execute();
                
                if ($stmt->rowCount() > 0) {
                    $row = $stmt->fetch(PDO::FETCH_ASSOC);
                    $feeSummary['session']['total'] = floatval($row['total']);
                    $feeSummary['session']['paid'] = floatval($row['paid']);
                    $feeSummary['session']['pending'] = $feeSummary['session']['total'] - $feeSummary['session']['paid'];
                    
                    if ($feeSummary['session']['total'] > 0) {
                        $feeSummary['session']['percentage'] = round(($feeSummary['session']['paid'] / $feeSummary['session']['total']) * 100, 2);
                    }
                }
                
                // Today's fee collection (from transactions)
                $query = "SELECT SUM(amount) as total 
                         FROM transactions 
                         WHERE student_fee_record_id IS NOT NULL 
                         AND date = :today 
                         AND is_cancelled = 0";
                $stmt = $this->conn->prepare($query);
                $stmt->bindParam(':today', $this->today);
                $stmt->execute();
                
                if ($stmt->rowCount() > 0) {
                    $row = $stmt->fetch(PDO::FETCH_ASSOC);
                    $feeSummary['today']['paid'] = floatval($row['total']);
                }
                
                // Month's fee collection
                $currentMonthStart = date('Y-m-01');
                $currentMonthEnd = date('Y-m-t');
                
                $query = "SELECT SUM(amount) as total 
                         FROM transactions 
                         WHERE student_fee_record_id IS NOT NULL 
                         AND date BETWEEN :start AND :end 
                         AND is_cancelled = 0";
                $stmt = $this->conn->prepare($query);
                $stmt->bindParam(':start', $currentMonthStart);
                $stmt->bindParam(':end', $currentMonthEnd);
                $stmt->execute();
                
                if ($stmt->rowCount() > 0) {
                    $row = $stmt->fetch(PDO::FETCH_ASSOC);
                    $feeSummary['month']['paid'] = floatval($row['total']);
                }
                
                // Get fee status breakdown
                $query = "SELECT status, COUNT(id) as count 
                         FROM student_fee_records 
                         WHERE student_record_id IN (
                            SELECT id FROM student_records WHERE academic_session_id = :academic_session_id
                         )
                         GROUP BY status";
                $stmt = $this->conn->prepare($query);
                $stmt->bindParam(':academic_session_id', $this->academicSessionId);
                $stmt->execute();
                
                if ($stmt->rowCount() > 0) {
                    while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
                        if ($row['status'] == 'paid') {
                            $feeSummary['status_breakdown']['paid'] = $row['count'];
                        } else if ($row['status'] == 'partially_paid') {
                            $feeSummary['status_breakdown']['partially_paid'] = $row['count'];
                        } else if ($row['status'] == 'pending') {
                            $feeSummary['status_breakdown']['pending'] = $row['count'];
                        } else if ($row['status'] == 'overdue') {
                            $feeSummary['status_breakdown']['overdue'] = $row['count'];
                        }
                    }
                }
                
                return $feeSummary;
            } catch(PDOException $e) {
                throw new Exception("Error fetching fee collection summary: " . $e->getMessage());
            }
        }
        
        // Get financial summary (income vs expense)
        public function getFinancialSummary() {
            try {
                $financialSummary = [
                    'today' => [
                        'income' => 0,
                        'expense' => 0,
                        'balance' => 0
                    ],
                    'month' => [
                        'income' => 0,
                        'expense' => 0,
                        'balance' => 0
                    ],
                    'year' => [
                        'income' => 0,
                        'expense' => 0,
                        'balance' => 0
                    ],
                    'monthly_trend' => [],
                    'category_breakdown' => [
                        'income' => [],
                        'expense' => []
                    ]
                ];
                
                // Today's income and expense
                $query = "SELECT SUM(amount) as total FROM incomes WHERE date_of_income = :today AND is_cancelled = 0";
                $stmt = $this->conn->prepare($query);
                $stmt->bindParam(':today', $this->today);
                $stmt->execute();
                
                if ($stmt->rowCount() > 0) {
                    $row = $stmt->fetch(PDO::FETCH_ASSOC);
                    $financialSummary['today']['income'] = floatval($row['total']);
                }
                
                $query = "SELECT SUM(amount) as total FROM expenses WHERE date_of_expense = :today AND is_cancelled = 0";
                $stmt = $this->conn->prepare($query);
                $stmt->bindParam(':today', $this->today);
                $stmt->execute();
                
                if ($stmt->rowCount() > 0) {
                    $row = $stmt->fetch(PDO::FETCH_ASSOC);
                    $financialSummary['today']['expense'] = floatval($row['total']);
                }
                
                $financialSummary['today']['balance'] = $financialSummary['today']['income'] - $financialSummary['today']['expense'];
                
                // Month's income and expense
                $currentMonthStart = date('Y-m-01');
                $currentMonthEnd = date('Y-m-t');
                
                $query = "SELECT SUM(amount) as total FROM incomes WHERE date_of_income BETWEEN :start AND :end AND is_cancelled = 0";
                $stmt = $this->conn->prepare($query);
                $stmt->bindParam(':start', $currentMonthStart);
                $stmt->bindParam(':end', $currentMonthEnd);
                $stmt->execute();
                
                if ($stmt->rowCount() > 0) {
                    $row = $stmt->fetch(PDO::FETCH_ASSOC);
                    $financialSummary['month']['income'] = floatval($row['total']);
                }
                
                $query = "SELECT SUM(amount) as total FROM expenses WHERE date_of_expense BETWEEN :start AND :end AND is_cancelled = 0";
                $stmt = $this->conn->prepare($query);
                $stmt->bindParam(':start', $currentMonthStart);
                $stmt->bindParam(':end', $currentMonthEnd);
                $stmt->execute();
                
                if ($stmt->rowCount() > 0) {
                    $row = $stmt->fetch(PDO::FETCH_ASSOC);
                    $financialSummary['month']['expense'] = floatval($row['total']);
                }
                
                $financialSummary['month']['balance'] = $financialSummary['month']['income'] - $financialSummary['month']['expense'];
                
                // Year's income and expense
                $currentYearStart = date('Y-01-01');
                $currentYearEnd = date('Y-12-31');
                
                $query = "SELECT SUM(amount) as total FROM incomes WHERE date_of_income BETWEEN :start AND :end AND is_cancelled = 0";
                $stmt = $this->conn->prepare($query);
                $stmt->bindParam(':start', $currentYearStart);
                $stmt->bindParam(':end', $currentYearEnd);
                $stmt->execute();
                
                if ($stmt->rowCount() > 0) {
                    $row = $stmt->fetch(PDO::FETCH_ASSOC);
                    $financialSummary['year']['income'] = floatval($row['total']);
                }
                
                $query = "SELECT SUM(amount) as total FROM expenses WHERE date_of_expense BETWEEN :start AND :end AND is_cancelled = 0";
                $stmt = $this->conn->prepare($query);
                $stmt->bindParam(':start', $currentYearStart);
                $stmt->bindParam(':end', $currentYearEnd);
                $stmt->execute();
                
                if ($stmt->rowCount() > 0) {
                    $row = $stmt->fetch(PDO::FETCH_ASSOC);
                    $financialSummary['year']['expense'] = floatval($row['total']);
                }
                
                $financialSummary['year']['balance'] = $financialSummary['year']['income'] - $financialSummary['year']['expense'];
                
                // Monthly trend for current year
                for ($i = 1; $i <= 12; $i++) {
                    $monthStart = date('Y-' . str_pad($i, 2, '0', STR_PAD_LEFT) . '-01');
                    $monthEnd = date('Y-' . str_pad($i, 2, '0', STR_PAD_LEFT) . '-t', strtotime($monthStart));
                    
                    $monthData = [
                        'month' => date('M', strtotime($monthStart)),
                        'income' => 0,
                        'expense' => 0,
                        'balance' => 0
                    ];
                    
                    $query = "SELECT SUM(amount) as total FROM incomes WHERE date_of_income BETWEEN :start AND :end AND is_cancelled = 0";
                    $stmt = $this->conn->prepare($query);
                    $stmt->bindParam(':start', $monthStart);
                    $stmt->bindParam(':end', $monthEnd);
                    $stmt->execute();
                    
                    if ($stmt->rowCount() > 0) {
                        $row = $stmt->fetch(PDO::FETCH_ASSOC);
                        $monthData['income'] = floatval($row['total']);
                    }
                    
                    $query = "SELECT SUM(amount) as total FROM expenses WHERE date_of_expense BETWEEN :start AND :end AND is_cancelled = 0";
                    $stmt = $this->conn->prepare($query);
                    $stmt->bindParam(':start', $monthStart);
                    $stmt->bindParam(':end', $monthEnd);
                    $stmt->execute();
                    
                    if ($stmt->rowCount() > 0) {
                        $row = $stmt->fetch(PDO::FETCH_ASSOC);
                        $monthData['expense'] = floatval($row['total']);
                    }
                    
                    $monthData['balance'] = $monthData['income'] - $monthData['expense'];
                    
                    $financialSummary['monthly_trend'][] = $monthData;
                }
                
                // Category breakdown for current month
                $query = "SELECT tc.name, SUM(i.amount) as total 
                        FROM incomes i 
                        JOIN transaction_categories tc ON i.transaction_category_id = tc.id 
                        WHERE i.date_of_income BETWEEN :start AND :end 
                        AND i.is_cancelled = 0 
                        GROUP BY tc.name 
                        ORDER BY total DESC 
                        LIMIT 5";
                $stmt = $this->conn->prepare($query);
                $stmt->bindParam(':start', $currentMonthStart);
                $stmt->bindParam(':end', $currentMonthEnd);
                $stmt->execute();
                
                if ($stmt->rowCount() > 0) {
                    while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
                        $financialSummary['category_breakdown']['income'][] = [
                            'category' => $row['name'],
                            'amount' => floatval($row['total'])
                        ];
                    }
                }
                
                $query = "SELECT tc.name, SUM(e.amount) as total 
                        FROM expenses e 
                        JOIN transaction_categories tc ON e.transaction_category_id = tc.id 
                        WHERE e.date_of_expense BETWEEN :start AND :end 
                        AND e.is_cancelled = 0 
                        GROUP BY tc.name 
                        ORDER BY total DESC 
                        LIMIT 5";
                $stmt = $this->conn->prepare($query);
                $stmt->bindParam(':start', $currentMonthStart);
                $stmt->bindParam(':end', $currentMonthEnd);
                $stmt->execute();
                
                if ($stmt->rowCount() > 0) {
                    while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
                        $financialSummary['category_breakdown']['expense'][] = [
                            'category' => $row['name'],
                            'amount' => floatval($row['total'])
                        ];
                    }
                }
                
                return $financialSummary;
            } catch(PDOException $e) {
                throw new Exception("Error fetching financial summary: " . $e->getMessage());
            }
        }
        
        // Get student statistics
        public function getStudentStatistics() {
            try {
                $studentStats = [
                    'total_students' => 0,
                    'new_admissions' => 0,
                    'gender_distribution' => [
                        'male' => 0,
                        'female' => 0,
                        'other' => 0
                    ],
                    'course_distribution' => [],
                    'batch_distribution' => []
                ];
                
                // Total students in current session
                $query = "SELECT COUNT(id) as total FROM student_records WHERE academic_session_id = :academic_session_id AND date_of_exit IS NULL";
                $stmt = $this->conn->prepare($query);
                $stmt->bindParam(':academic_session_id', $this->academicSessionId);
                $stmt->execute();
                
                if ($stmt->rowCount() > 0) {
                    $row = $stmt->fetch(PDO::FETCH_ASSOC);
                    $studentStats['total_students'] = $row['total'];
                }
                
                // New admissions in current month
                $currentMonthStart = date('Y-m-01');
                $currentMonthEnd = date('Y-m-t');
                
                $query = "SELECT COUNT(id) as total FROM student_records WHERE academic_session_id = :academic_session_id AND date_of_entry BETWEEN :start AND :end";
                $stmt = $this->conn->prepare($query);
                $stmt->bindParam(':academic_session_id', $this->academicSessionId);
                $stmt->bindParam(':start', $currentMonthStart);
                $stmt->bindParam(':end', $currentMonthEnd);
                $stmt->execute();
                
                if ($stmt->rowCount() > 0) {
                    $row = $stmt->fetch(PDO::FETCH_ASSOC);
                    $studentStats['new_admissions'] = $row['total'];
                }
                
                // Gender distribution
                $query = "SELECT s.gender, COUNT(sr.id) as count 
                        FROM student_records sr 
                        JOIN students s ON sr.student_id = s.id 
                        WHERE sr.academic_session_id = :academic_session_id 
                        AND sr.date_of_exit IS NULL 
                        GROUP BY s.gender";
                $stmt = $this->conn->prepare($query);
                $stmt->bindParam(':academic_session_id', $this->academicSessionId);
                $stmt->execute();
                
                if ($stmt->rowCount() > 0) {
                    while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
                        $gender = strtolower($row['gender']);
                        if ($gender == 'male' || $gender == 'm') {
                            $studentStats['gender_distribution']['male'] = $row['count'];
                        } else if ($gender == 'female' || $gender == 'f') {
                            $studentStats['gender_distribution']['female'] = $row['count'];
                        } else {
                            $studentStats['gender_distribution']['other'] += $row['count'];
                        }
                    }
                }
                
                // Course distribution
                $query = "SELECT c.name, COUNT(sr.id) as count 
                        FROM student_records sr 
                        JOIN batches b ON sr.batch_id = b.id 
                        JOIN courses c ON b.course_id = c.id 
                        WHERE sr.academic_session_id = :academic_session_id 
                        AND sr.date_of_exit IS NULL 
                        GROUP BY c.name 
                        ORDER BY count DESC";
                $stmt = $this->conn->prepare($query);
                $stmt->bindParam(':academic_session_id', $this->academicSessionId);
                $stmt->execute();
                
                if ($stmt->rowCount() > 0) {
                    while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
                        $studentStats['course_distribution'][] = [
                            'course' => $row['name'],
                            'count' => $row['count']
                        ];
                    }
                }
                
                // Batch distribution
                $query = "SELECT b.name, COUNT(sr.id) as count 
                        FROM student_records sr 
                        JOIN batches b ON sr.batch_id = b.id 
                        WHERE sr.academic_session_id = :academic_session_id 
                        AND sr.date_of_exit IS NULL 
                        GROUP BY b.name 
                        ORDER BY count DESC";
                $stmt = $this->conn->prepare($query);
                $stmt->bindParam(':academic_session_id', $this->academicSessionId);
                $stmt->execute();
                
                if ($stmt->rowCount() > 0) {
                    while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
                        $studentStats['batch_distribution'][] = [
                            'batch' => $row['name'],
                            'count' => $row['count']
                        ];
                    }
                }
                
                return $studentStats;
            } catch(PDOException $e) {
                throw new Exception("Error fetching student statistics: " . $e->getMessage());
            }
        }
        
        // Get academic calendar events
        public function getUpcomingEvents() {
            try {
                $upcomingEvents = [
                    'meetings' => [],
                    'events' => [],
                    'exams' => [],
                    'assignments' => [],
                    'homework' => []
                ];
                
                // Upcoming meetings
                $query = "SELECT id, title, description, date, start_time, end_time 
                        FROM meetings 
                        WHERE date >= :today 
                        ORDER BY date, start_time 
                        LIMIT 5";
                $stmt = $this->conn->prepare($query);
                $stmt->bindParam(':today', $this->today);
                $stmt->execute();
                
                if ($stmt->rowCount() > 0) {
                    while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
                        $upcomingEvents['meetings'][] = [
                            'id' => $row['id'],
                            'title' => $row['title'],
                            'description' => $row['description'],
                            'date' => $row['date'],
                            'start_time' => $row['start_time'],
                            'end_time' => $row['end_time']
                        ];
                    }
                }
                
                // Upcoming events
                $query = "SELECT id, title, description, start_date, end_date, start_time, end_time, venue 
                        FROM events 
                        WHERE start_date >= :today 
                        ORDER BY start_date, start_time 
                        LIMIT 5";
                $stmt = $this->conn->prepare($query);
                $stmt->bindParam(':today', $this->today);
                $stmt->execute();
                
                if ($stmt->rowCount() > 0) {
                    while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
                        $upcomingEvents['events'][] = [
                            'id' => $row['id'],
                            'title' => $row['title'],
                            'description' => $row['description'],
                            'start_date' => $row['start_date'],
                            'end_date' => $row['end_date'],
                            'start_time' => $row['start_time'],
                            'end_time' => $row['end_time'],
                            'venue' => $row['venue']
                        ];
                    }
                }
                
                // Upcoming exams
                $query = "SELECT er.id, e.name as exam_name, s.name as subject_name, er.date, er.start 
                        FROM exam_records er 
                        JOIN exam_schedules es ON er.exam_schedule_id = es.id 
                        JOIN exams e ON es.exam_id = e.id 
                        JOIN subjects s ON er.subject_id = s.id 
                        WHERE er.date >= :today 
                        ORDER BY er.date, er.start 
                        LIMIT 5";
                $stmt = $this->conn->prepare($query);
                $stmt->bindParam(':today', $this->today);
                $stmt->execute();
                
                if ($stmt->rowCount() > 0) {
                    while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
                        $upcomingEvents['exams'][] = [
                            'id' => $row['id'],
                            'exam_name' => $row['exam_name'],
                            'subject_name' => $row['subject_name'],
                            'date' => $row['date'],
                            'start_time' => $row['start']
                        ];
                    }
                }
                
                // Upcoming assignments
                $query = "SELECT id, title, description, date_of_assignment, due_date 
                        FROM assignments 
                        WHERE due_date >= :today 
                        ORDER BY due_date 
                        LIMIT 5";
                $stmt = $this->conn->prepare($query);
                $stmt->bindParam(':today', $this->today);
                $stmt->execute();
                
                if ($stmt->rowCount() > 0) {
                    while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
                        $upcomingEvents['assignments'][] = [
                            'id' => $row['id'],
                            'title' => $row['title'],
                            'description' => substr($row['description'], 0, 100) . (strlen($row['description']) > 100 ? '...' : ''),
                            'date_of_assignment' => $row['date_of_assignment'],
                            'due_date' => $row['due_date']
                        ];
                    }
                }
                
                // Upcoming homework
                $query = "SELECT id, title, description, date_of_home_work, due_date 
                        FROM home_works 
                        WHERE due_date >= :today 
                        ORDER BY due_date 
                        LIMIT 5";
                $stmt = $this->conn->prepare($query);
                $stmt->bindParam(':today', $this->today);
                $stmt->execute();
                
                if ($stmt->rowCount() > 0) {
                    while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
                        $upcomingEvents['homework'][] = [
                            'id' => $row['id'],
                            'title' => $row['title'],
                            'description' => substr($row['description'], 0, 100) . (strlen($row['description']) > 100 ? '...' : ''),
                            'date_of_home_work' => $row['date_of_home_work'],
                            'due_date' => $row['due_date']
                        ];
                    }
                }
                
                return $upcomingEvents;
            } catch(PDOException $e) {
                throw new Exception("Error fetching upcoming events: " . $e->getMessage());
            }
        }
        
        // Get birthday alerts
        public function getBirthdayAlerts() {
            try {
                $birthdays = [
                    'today' => [
                        'students' => [],
                        'employees' => []
                    ],
                    'upcoming' => [
                        'students' => [],
                        'employees' => []
                    ]
                ];
                
                // Today's student birthdays
                $query = "SELECT s.id, s.first_name, s.middle_name, s.last_name, s.date_of_birth, s.student_photo,
                          b.name as batch_name, sr.roll_number
                          FROM students s
                          JOIN student_records sr ON s.id = sr.student_id
                          JOIN batches b ON sr.batch_id = b.id
                          WHERE DAY(s.date_of_birth) = DAY(:today)
                          AND MONTH(s.date_of_birth) = MONTH(:today)
                          AND sr.academic_session_id = :academic_session_id
                          AND sr.date_of_exit IS NULL
                          ORDER BY s.first_name";
                $stmt = $this->conn->prepare($query);
                $stmt->bindParam(':today', $this->today);
                $stmt->bindParam(':academic_session_id', $this->academicSessionId);
                $stmt->execute();
                
                if ($stmt->rowCount() > 0) {
                    while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
                        $birthdays['today']['students'][] = [
                            'id' => $row['id'],
                            'name' => $row['first_name'] . ' ' . $row['middle_name'] . ' ' . $row['last_name'],
                            'date_of_birth' => $row['date_of_birth'],
                            'photo' => $row['student_photo'],
                            'batch' => $row['batch_name'],
                            'roll_number' => $row['roll_number']
                        ];
                    }
                }
                
                // Today's employee birthdays
                $query = "SELECT id, first_name, middle_name, last_name, date_of_birth, photo
                          FROM employees
                          WHERE DAY(date_of_birth) = DAY(:today)
                          AND MONTH(date_of_birth) = MONTH(:today)
                          ORDER BY first_name";
                $stmt = $this->conn->prepare($query);
                $stmt->bindParam(':today', $this->today);
                $stmt->execute();
                
                if ($stmt->rowCount() > 0) {
                    while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
                        $birthdays['today']['employees'][] = [
                            'id' => $row['id'],
                            'name' => $row['first_name'] . ' ' . $row['middle_name'] . ' ' . $row['last_name'],
                            'date_of_birth' => $row['date_of_birth'],
                            'photo' => $row['photo']
                        ];
                    }
                }
                
                // Upcoming student birthdays (next 7 days)
                $nextWeek = date('Y-m-d', strtotime('+7 days'));
                
                $query = "SELECT s.id, s.first_name, s.middle_name, s.last_name, s.date_of_birth, s.student_photo,
                          b.name as batch_name, sr.roll_number
                          FROM students s
                          JOIN student_records sr ON s.id = sr.student_id
                          JOIN batches b ON sr.batch_id = b.id
                          WHERE ((MONTH(s.date_of_birth) = MONTH(:today) AND DAY(s.date_of_birth) > DAY(:today))
                          OR (MONTH(s.date_of_birth) = MONTH(:nextWeek) AND DAY(s.date_of_birth) <= DAY(:nextWeek)))
                          AND sr.academic_session_id = :academic_session_id
                          AND sr.date_of_exit IS NULL
                          ORDER BY MONTH(s.date_of_birth), DAY(s.date_of_birth)
                          LIMIT 5";
                $stmt = $this->conn->prepare($query);
                $stmt->bindParam(':today', $this->today);
                $stmt->bindParam(':nextWeek', $nextWeek);
                $stmt->bindParam(':academic_session_id', $this->academicSessionId);
                $stmt->execute();
                
                if ($stmt->rowCount() > 0) {
                    while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
                        $birthdays['upcoming']['students'][] = [
                            'id' => $row['id'],
                            'name' => $row['first_name'] . ' ' . $row['middle_name'] . ' ' . $row['last_name'],
                            'date_of_birth' => $row['date_of_birth'],
                            'photo' => $row['student_photo'],
                            'batch' => $row['batch_name'],
                            'roll_number' => $row['roll_number']
                        ];
                    }
                }
                
                // Upcoming employee birthdays (next 7 days)
                $query = "SELECT id, first_name, middle_name, last_name, date_of_birth, photo
                          FROM employees
                          WHERE ((MONTH(date_of_birth) = MONTH(:today) AND DAY(date_of_birth) > DAY(:today))
                          OR (MONTH(date_of_birth) = MONTH(:nextWeek) AND DAY(date_of_birth) <= DAY(:nextWeek)))
                          ORDER BY MONTH(date_of_birth), DAY(date_of_birth)
                          LIMIT 5";
                $stmt = $this->conn->prepare($query);
                $stmt->bindParam(':today', $this->today);
                $stmt->bindParam(':nextWeek', $nextWeek);
                $stmt->execute();
                
                if ($stmt->rowCount() > 0) {
                    while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
                        $birthdays['upcoming']['employees'][] = [
                            'id' => $row['id'],
                            'name' => $row['first_name'] . ' ' . $row['middle_name'] . ' ' . $row['last_name'],
                            'date_of_birth' => $row['date_of_birth'],
                            'photo' => $row['photo']
                        ];
                    }
                }
                
                return $birthdays;
            } catch(PDOException $e) {
                throw new Exception("Error fetching birthday alerts: " . $e->getMessage());
            }
        }
        
        // Get library statistics
        public function getLibraryStatistics() {
            try {
                $libraryStats = [
                    'total_books' => 0,
                    'total_issued' => 0,
                    'total_available' => 0,
                    'overdue_books' => 0,
                    'recent_issues' => [],
                    'recent_returns' => []
                ];
                
                // Total books in library
                $query = "SELECT COUNT(id) as total FROM books";
                $stmt = $this->conn->prepare($query);
                $stmt->execute();
                
                if ($stmt->rowCount() > 0) {
                    $row = $stmt->fetch(PDO::FETCH_ASSOC);
                    $libraryStats['total_books'] = $row['total'];
                }
                
                // Total books issued
                $query = "SELECT COUNT(bl.id) as total_issued
                          FROM book_logs bl
                          WHERE bl.id NOT IN (
                              SELECT book_log_id FROM book_log_details WHERE date_of_return IS NOT NULL
                          )";
                $stmt = $this->conn->prepare($query);
                $stmt->execute();
                
                if ($stmt->rowCount() > 0) {
                    $row = $stmt->fetch(PDO::FETCH_ASSOC);
                    $libraryStats['total_issued'] = $row['total_issued'];
                }
                
                $libraryStats['total_available'] = $libraryStats['total_books'] - $libraryStats['total_issued'];
                
                // Overdue books
                $query = "SELECT COUNT(bl.id) as overdue
                          FROM book_logs bl
                          WHERE bl.due_date < :today
                          AND bl.id NOT IN (
                              SELECT book_log_id FROM book_log_details WHERE date_of_return IS NOT NULL
                          )";
                $stmt = $this->conn->prepare($query);
                $stmt->bindParam(':today', $this->today);
                $stmt->execute();
                
                if ($stmt->rowCount() > 0) {
                    $row = $stmt->fetch(PDO::FETCH_ASSOC);
                    $libraryStats['overdue_books'] = $row['overdue'];
                }
                
                // Recent book issues
                $query = "SELECT bl.id, b.title as book_title, 
                          CONCAT(s.first_name, ' ', s.last_name) as student_name,
                          bl.date_of_issue, bl.due_date
                          FROM book_logs bl
                          JOIN book_log_details bld ON bl.id = bld.book_log_id
                          JOIN book_post_details bpd ON bld.book_post_detail_id = bpd.id
                          JOIN book_posts bp ON bpd.book_post_id = bp.id
                          JOIN books b ON bp.book_id = b.id
                          JOIN student_records sr ON bl.student_record_id = sr.id
                          JOIN students s ON sr.student_id = s.id
                          ORDER BY bl.date_of_issue DESC
                          LIMIT 5";
                $stmt = $this->conn->prepare($query);
                $stmt->execute();
                
                if ($stmt->rowCount() > 0) {
                    while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
                        $libraryStats['recent_issues'][] = [
                            'id' => $row['id'],
                            'book_title' => $row['book_title'],
                            'student_name' => $row['student_name'],
                            'date_of_issue' => $row['date_of_issue'],
                            'due_date' => $row['due_date']
                        ];
                    }
                }
                
                // Recent book returns
                $query = "SELECT bl.id, b.title as book_title, 
                          CONCAT(s.first_name, ' ', s.last_name) as student_name,
                          bl.date_of_issue, bld.date_of_return
                          FROM book_logs bl
                          JOIN book_log_details bld ON bl.id = bld.book_log_id
                          JOIN book_post_details bpd ON bld.book_post_detail_id = bpd.id
                          JOIN book_posts bp ON bpd.book_post_id = bp.id
                          JOIN books b ON bp.book_id = b.id
                          JOIN student_records sr ON bl.student_record_id = sr.id
                          JOIN students s ON sr.student_id = s.id
                          WHERE bld.date_of_return IS NOT NULL
                          ORDER BY bld.date_of_return DESC
                          LIMIT 5";
                $stmt = $this->conn->prepare($query);
                $stmt->execute();
                
                if ($stmt->rowCount() > 0) {
                    while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
                        $libraryStats['recent_returns'][] = [
                            'id' => $row['id'],
                            'book_title' => $row['book_title'],
                            'student_name' => $row['student_name'],
                            'date_of_issue' => $row['date_of_issue'],
                            'date_of_return' => $row['date_of_return']
                        ];
                    }
                }
                
                return $libraryStats;
            } catch(PDOException $e) {
                throw new Exception("Error fetching library statistics: " . $e->getMessage());
            }
        }
        
        // Get recent activities
        public function getRecentActivities() {
            try {
                $recentActivities = [];
                
                $query = "SELECT id, log_name, description, created_at
                          FROM activity_log
                          ORDER BY created_at DESC
                          LIMIT 10";
                $stmt = $this->conn->prepare($query);
                $stmt->execute();
                
                if ($stmt->rowCount() > 0) {
                    while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
                        $recentActivities[] = [
                            'id' => $row['id'],
                            'type' => $row['log_name'],
                            'description' => $row['description'],
                            'timestamp' => $row['created_at']
                        ];
                    }
                }
                
                return $recentActivities;
            } catch(PDOException $e) {
                throw new Exception("Error fetching recent activities: " . $e->getMessage());
            }
        }
        
        // Get complete dashboard summary
        public function getDashboardSummary() {
            return [
                'attendance' => $this->getAttendanceSummary(),
                'fee_collection' => $this->getFeeCollectionSummary(),
                'financial' => $this->getFinancialSummary(),
                'students' => $this->getStudentStatistics(),
                'upcoming_events' => $this->getUpcomingEvents(),
                'birthdays' => $this->getBirthdayAlerts(),
                'library' => $this->getLibraryStatistics(),
                'recent_activities' => $this->getRecentActivities()
            ];
        }
    }

    // Initialize Dashboard object
    $dashboard = new DashboardSummary($db);

    // Get request parameters
    $endpoint = isset($_GET['endpoint']) ? $_GET['endpoint'] : 'summary';

    // Get data based on parameters
    $response = array();

    switch ($endpoint) {
        case 'attendance':
            $response = $dashboard->getAttendanceSummary();
            break;
        case 'fee_collection':
            $response = $dashboard->getFeeCollectionSummary();
            break;
        case 'financial':
            $response = $dashboard->getFinancialSummary();
            break;
        case 'students':
            $response = $dashboard->getStudentStatistics();
            break;
        case 'events':
            $response = $dashboard->getUpcomingEvents();
            break;
        case 'birthdays':
            $response = $dashboard->getBirthdayAlerts();
            break;
        case 'library':
            $response = $dashboard->getLibraryStatistics();
            break;
        case 'activities':
            $response = $dashboard->getRecentActivities();
            break;
        case 'summary':
        default:
            $response = $dashboard->getDashboardSummary();
            break;
    }

    http_response_code(200);
    echo json_encode(array(
        "status" => true,
        "message" => "Dashboard data retrieved successfully",
        "data" => $response
    ));

} catch(Exception $e) {
    http_response_code(500);
    echo json_encode(array(
        "status" => false,
        "message" => "Server error occurred",
        "error" => $e->getMessage(),
        "trace" => debug_backtrace()
    ));
}
?>
                        